import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Execute Oracle Plan ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Rotating gripper to ninety_deg")
        # Quaternion for 90 degrees about Z: [x, y, z, w]
        target_quat_ninety = np.array([0.0, 0.0, np.sqrt(2)/2, np.sqrt(2)/2])
        try:
            obs, reward, done = rotate(env, task, target_quat=target_quat_ninety)
        except Exception as e:
            print(f"[Task] Error during rotate: {e}")
            return
        if done:
            print("[Task] Task ended after rotate. Reward:", reward)
            return

        # Step 2: Move to side position of bottom drawer
        print("[Task] Moving to bottom side position")
        if 'bottom_side_pos' not in positions:
            print("[Task] Error: 'bottom_side_pos' not found in positions.")
            return
        side_pos_bottom = np.array(positions['bottom_side_pos'])
        try:
            obs, reward, done = move(env, task, target_pos=side_pos_bottom)
        except Exception as e:
            print(f"[Task] Error during move to side position: {e}")
            return
        if done:
            print("[Task] Task ended after moving to side position. Reward:", reward)
            return

        # Step 3: Move to anchor position of bottom drawer
        print("[Task] Moving to bottom anchor position")
        if 'bottom_anchor_pos' not in positions:
            print("[Task] Error: 'bottom_anchor_pos' not found in positions.")
            return
        anchor_pos_bottom = np.array(positions['bottom_anchor_pos'])
        try:
            obs, reward, done = move(env, task, target_pos=anchor_pos_bottom)
        except Exception as e:
            print(f"[Task] Error during move to anchor position: {e}")
            return
        if done:
            print("[Task] Task ended after moving to anchor position. Reward:", reward)
            return

        # Step 4: Pick the bottom drawer handle at anchor position
        print("[Task] Picking bottom drawer handle")
        try:
            obs, reward, done = pick(env, task, target_pos=anchor_pos_bottom)
        except Exception as e:
            print(f"[Task] Error during pick: {e}")
            return
        if done:
            print("[Task] Task ended after pick. Reward:", reward)
            return

        # Step 5: Pull the drawer to open it (unlock)
        print("[Task] Pulling bottom drawer to open")
        pull_distance = 0.2  # meters
        pull_axis = 'x'
        try:
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        except Exception as e:
            print(f"[Task] Error during pull: {e}")
            return
        if done:
            print("[Task] Task ended after pull. Reward:", reward)
            return

        # Final check
        print("[Task] Finished unlocking the cabinet by opening bottom drawer. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()