import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing the simulation
        init_video_writers(obs)

        # Wrap step and get_observation for recording/debugging
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_pos = np.array(positions['bottom_side_pos'])
        anchor_pos = np.array(positions['bottom_anchor_pos'])

        # === Prepare Plan Parameters ===
        # Rotate gripper to 90° about Z
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()

        print("[Plan] Executing oracle plan...")
        
        # === Oracle Plan Execution ===
        # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, side_pos)
        obs, reward, done = move(env, task, anchor_pos)
        obs, reward, done = pick(env, task, anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        # Frozen Code End

        # Check final status
        if done:
            print("[Task] Plan completed. Reward:", reward)
        else:
            print("[Task] Plan ended (done=False). Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()