import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the disposal task: put rubbish in bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap task.step and task.get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We want to move 'rubbish' into 'bin'
        item1_pos = positions.get('rubbish')
        bin_pos   = positions.get('bin')

        if item1_pos is None or bin_pos is None:
            raise ValueError("Could not find required object positions for 'rubbish' or 'bin'")

        # === Oracle Plan Execution ===
        try:
            # Step 1: pick 'rubbish' at its current location
            print("[Task] Step 1: Picking 'rubbish' at position:", item1_pos)
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(item1_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Episode ended during pick. Reward:", reward)
                return

            # Step 2: place 'rubbish' into 'bin'
            print("[Task] Step 2: Placing 'rubbish' into 'bin' at position:", bin_pos)
            obs, reward, done = place(
                env,
                task,
                target_pos=np.array(bin_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False). Reward:", reward)

        except Exception as e:
            print("[Task] Exception during plan execution:", e)

    finally:
        # Always ensure environment is shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()