import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Concrete task: pick the rubbish from the table and put it in the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        required_keys = ['rubbish', 'table', 'bin']
        for key in required_keys:
            if key not in positions:
                raise KeyError(f"Position for '{key}' not found in object_positions")
        rubbish_pos = np.array(positions['rubbish'])
        bin_pos     = np.array(positions['bin'])
        # Note: 'table' position is known but not needed explicitly because pick uses the object pos

        # === Execute Oracle Plan ===

        # Step 1: pick rubbish from table
        print("[Task] Step 1: Picking rubbish")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after pick. Reward:", reward)
            return

        # Step 2: place rubbish into bin
        print("[Task] Step 2: Placing rubbish into bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Final reward:", reward)
        else:
            print("[Task] Finished all steps but done flag is False. Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        # Ensure the environment is always shut down
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()