import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Extract positions for the plan's objects
        try:
            rubbish_pos = positions['rubbish']
            bin_pos = positions['bin']
        except KeyError as e:
            raise RuntimeError(f"Expected object not found in positions: {e}")

        # === Execute Oracle Plan ===
        # Step 1: Pick up the rubbish from the table
        print("[Task] Step 1: Picking up 'rubbish' from its location at:", rubbish_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(rubbish_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended during pick. Reward:", reward)
            return

        # Step 2: Place the rubbish into the bin
        print("[Task] Step 2: Placing 'rubbish' into the 'bin' at:", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(bin_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task ended without done=True. Current reward:", reward)

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
        raise
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()