import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        table_pos = positions['table']
        bin_pos = positions['bin']
        rubbish_pos = positions['item1']  # Assuming item1 is the rubbish

        # Step 1: Rotate the gripper to the correct angle
        print("[Task] Rotating gripper...")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Gripper rotated successfully!")
        else:
            print("[Task] Failed to rotate the gripper.")

        # Step 2: Move to the side position of the drawer
        print("[Task] Moving to side position of the drawer...")
        obs, reward, done = move_to_side(env, task, 'gripper', 'bottom', 'nowhere_pos', 'bottom_side_pos')
        if done:
            print("[Task] Moved to side position successfully!")
        else:
            print("[Task] Failed to move to side position.")

        # Step 3: Move to the anchor position of the drawer
        print("[Task] Moving to anchor position of the drawer...")
        obs, reward, done = move_to_anchor(env, task, 'gripper', 'bottom', 'bottom_side_pos', 'bottom_anchor_pos')
        if done:
            print("[Task] Moved to anchor position successfully!")
        else:
            print("[Task] Failed to move to anchor position.")

        # Step 4: Pick the drawer
        print("[Task] Picking the drawer...")
        obs, reward, done = pick_drawer(env, task, 'gripper', 'bottom', 'bottom_anchor_pos')
        if done:
            print("[Task] Drawer picked successfully!")
        else:
            print("[Task] Failed to pick the drawer.")

        # Step 5: Pull the drawer to open it
        print("[Task] Pulling the drawer to open...")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Drawer opened successfully!")
        else:
            print("[Task] Failed to open the drawer.")

        # Step 6: Pick up the rubbish
        print("[Task] Picking up the rubbish...")
        obs, reward, done = pick(env, task, rubbish_pos, table_pos)
        if done:
            print("[Task] Rubbish picked successfully!")
        else:
            print("[Task] Failed to pick up the rubbish.")

        # Step 7: Place the rubbish in the bin
        print("[Task] Placing rubbish in the bin...")
        obs, reward, done = place(env, task, rubbish_pos, bin_pos)
        if done:
            print("[Task] Rubbish placed in the bin successfully!")
        else:
            print("[Task] Failed to place rubbish in the bin.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()