import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *  # Importing predefined skills
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        plate_pos = positions['plate']
        tomato1_pos = positions['item1']
        tomato2_pos = positions['item2']
        drawer_pos = positions['bottom_anchor_pos']  # Assuming we want to pull the bottom drawer

        # Step 1: Rotate the gripper to the correct orientation (if needed)
        print("[Task] Rotating gripper to the correct orientation.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Assuming no rotation needed
        if done:
            print("[Task] Gripper rotated successfully!")
        else:
            print("[Task] Failed to rotate the gripper.")
            return

        # Step 2: Move to the side position of the drawer
        print("[Task] Moving to the side position of the drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Moved to side position successfully!")
        else:
            print("[Task] Failed to move to side position.")
            return

        # Step 3: Move to the anchor position of the drawer
        print("[Task] Moving to the anchor position of the drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Moved to anchor position successfully!")
        else:
            print("[Task] Failed to move to anchor position.")
            return

        # Step 4: Pick the drawer
        print("[Task] Picking the drawer.")
        obs, reward, done = pick(env, task, target_pos=drawer_pos)
        if done:
            print("[Task] Drawer picked successfully!")
        else:
            print("[Task] Failed to pick the drawer.")
            return

        # Step 5: Pull open the drawer
        print("[Task] Pulling open the drawer.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Drawer opened successfully!")
        else:
            print("[Task] Failed to open the drawer.")
            return

        # Step 6: Pick the first tomato
        print("[Task] Picking tomato1.")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Tomato1 picked successfully!")
        else:
            print("[Task] Failed to pick tomato1.")
            return

        # Step 7: Place the first tomato onto the plate
        print("[Task] Placing tomato1 onto the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Tomato1 placed successfully!")
        else:
            print("[Task] Failed to place tomato1.")
            return

        # Step 8: Pick the second tomato
        print("[Task] Picking tomato2.")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Tomato2 picked successfully!")
        else:
            print("[Task] Failed to pick tomato2.")
            return

        # Step 9: Place the second tomato onto the plate
        print("[Task] Placing tomato2 onto the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Tomato2 placed successfully!")
        else:
            print("[Task] Failed to place tomato2.")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()