import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define the drawer and tomatoes
        bottom_drawer = 'bottom'
        tomato1 = 'item1'
        tomato2 = 'item2'
        plate = 'plate'

        # Step 1: Pull the bottom drawer to unlock the cabinet
        print("[Task] Pulling the bottom drawer to unlock the cabinet.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return

        # Step 2: Rotate the gripper to the correct orientation
        print("[Task] Rotating the gripper to the correct orientation.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 0, 1]))  # Assuming no rotation needed
        if done:
            print("[Task] Task ended after rotating the gripper!")
            return

        # Step 3: Move to the side position of the bottom drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Task ended after moving to the side position!")
            return

        # Step 4: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Task ended after moving to the anchor position!")
            return

        # Step 5: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=positions['bottom_anchor_pos'], loc=bottom_drawer)
        if done:
            print("[Task] Task ended after picking the bottom drawer!")
            return

        # Step 6: Pull the bottom drawer to unlock the cabinet
        print("[Task] Pulling the bottom drawer to unlock the cabinet.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')
        if done:
            print("[Task] Task ended after pulling the drawer!")
            return

        # Step 7: Pick the first tomato from the table
        print("[Task] Picking the first tomato from the table.")
        obs, reward, done = pick(env, task, target_pos=positions[tomato1], loc='table')
        if done:
            print("[Task] Task ended after picking the first tomato!")
            return

        # Step 8: Place the first tomato onto the plate
        print("[Task] Placing the first tomato onto the plate.")
        obs, reward, done = place(env, task, target_pos=positions[plate], loc='plate')
        if done:
            print("[Task] Task ended after placing the first tomato!")
            return

        # Step 9: Pick the second tomato from the table
        print("[Task] Picking the second tomato from the table.")
        obs, reward, done = pick(env, task, target_pos=positions[tomato2], loc='table')
        if done:
            print("[Task] Task ended after picking the second tomato!")
            return

        # Step 10: Place the second tomato onto the plate
        print("[Task] Placing the second tomato onto the plate.")
        obs, reward, done = place(env, task, target_pos=positions[plate], loc='plate')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()