import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']
        middle_anchor_pos = positions['middle_anchor_pos']
        middle_side_pos = positions['middle_side_pos']

        # === Execute the Plan ===
        # Step 1: Rotate the gripper to ninety degrees
        rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)

        # Step 2: Move to the side position of the middle drawer
        move(env, task, target_pos=middle_side_pos, max_steps=100, threshold=0.01, timeout=10.0)

        # Step 3: Move to the anchor position of the middle drawer
        move(env, task, target_pos=middle_anchor_pos, max_steps=100, threshold=0.01, timeout=10.0)

        # Step 4: Pick the middle drawer
        pick(env, task, 'gripper', 'middle', 'anchor-pos-middle')

        # Step 5: Pull the middle drawer to open it
        pull(env, task, pull_distance=0.1, pull_axis='x', max_steps=100, threshold=0.01, timeout=10.0)

        # Step 6: Pick the first tomato
        pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)

        # Step 7: Place the first tomato on the plate
        place(env, task, target_pos=plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)

        # Step 8: Pick the second tomato
        pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)

        # Step 9: Place the second tomato on the plate
        place(env, task, target_pos=plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()