# run_skeleton_task.py (Completed According to Requirements)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # ONLY use predefined skills; do NOT define new ones

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        positions = get_object_positions()
        #
        # ---------- EXPLORATION PHASE ----------
        #
        # Per feedback, we are missing knowledge about (drawer-closed drawer_bottom)!
        # We need to explore to determine if 'drawer_bottom' is actually closed.
        #
        # We use the relevant skill to explore the property of 'drawer_bottom':
        # Let's try to PULL the drawer. If it fails, maybe it is not closed/unlocked.
        # Name convention in domain: 'execute_pull' (drawer, handle, location), requires knowledge
        # Ensure we gracefully check for existence.
        #
        # For exploration, we want to know if the predicate (drawer-closed drawer_bottom)
        # holds, i.e., discover the physical state of the drawer.
        
        # Standard object naming assumptions:
        drawer_name = 'drawer_bottom'
        handle_name = None
        robot_location = None
        
        # Try to find the handle object and the robot location from positions.
        # Fall back to default names if not available.
        for name in positions:
            if 'handle' in name and 'bottom' in name:
                handle_name = name
        # Default: If not found, use 'handle_bottom'
        if handle_name is None:
            handle_name = 'handle_bottom'
        
        # Find robot's current location
        if 'robot' in positions:
            robot_location = positions['robot']
        else:
            # If not specified, we use a location-value or default string/location.
            robot_location = 'ready-pose'
        
        print("[Exploration] Trying to pull the drawer to determine if it is closed or locked.")
        # We need to ensure the robot holds the handle object before pulling.
        # To do that, we execute:
        # 1. Move to the handle's location (if position known)
        # 2. Pick up the handle using execute_pick
        # 3. Try to pull the drawer with execute_pull to check lock/closed status
        
        # 1. "Navigate" to handle location (if positions allows)
        handle_pos = positions.get(handle_name, None)
        if handle_pos is not None and hasattr(handle_pos, '__iter__'):
            # If a 3D position, use execute_go (from location to location)
            print(f"[Exploration] Moving to handle location: {handle_name} at {handle_pos}")
            try:
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=handle_pos)
                robot_location = handle_pos
            except Exception as e:
                print(f"[Exploration] Could not go to handle: {e}")
        else:
            print("[Exploration] No usable handle position information.")
        
        # 2. Pick the handle with execute_pick (object, location)
        print(f"[Exploration] Attempting to pick handle: {handle_name}")
        try:
            obs, reward, done = execute_pick(env, task, object_name=handle_name, location=robot_location)
            if done:
                print("[Exploration] Successfully picked the handle.")
            else:
                print("[Exploration] Failed to pick the handle.")
        except Exception as e:
            print(f"[Exploration] Exception during pick handle: {e}")
        
        # 3. Try to pull the drawer (drawer, handle, location)
        print(f"[Exploration] Attempting to pull the drawer: {drawer_name} with handle {handle_name} at {robot_location}")
        try:
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_location)
            if done:
                print("[Exploration] Drawer successfully pulled. Not locked and was closed.")
            else:
                print("[Exploration] Drawer could not be pulled. Possibly not closed or locked.")
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
        #
        # Now the predicate (drawer-closed drawer_bottom) should be known (either confirmed or disproved)
        # Documenting exploration result.
        print("[Exploration] Exploration of (drawer-closed drawer_bottom) completed.")
        
        # -----------------------------------------------
        # Continue with the rest of the plan (oracle plan)
        # As per instructions, this is just the exploration portion for missing predicate;
        # add core logic for task plan below, using only the provided skills as needed
        # Example structure:
        #
        # print("[Task] Performing next oracle plan steps...")
        # obs, reward, done = execute_go(env, task, ...)
        # obs, reward, done = execute_pick(env, task, ...)
        # obs, reward, done = execute_place(env, task, ...)
        # if done:
        #     print("[Task] Task completed!")
        #     return
        #
        # For this task, only the exploration for (drawer-closed drawer_bottom) per feedback is required
        
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
