# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from corrected object list) ===
        dice_list = ['dice1', 'dice2']
        handle_list = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_list = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        location_list = ['floor', 'drawer-area', 'ready-pose']

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we need to check for missing predicates
        # such as lock-known, identified, temperature-known, etc.
        # We'll use available skills to explore and try to discover the missing predicate.

        # For this environment, let's try to pull each drawer handle and see if any action fails,
        # indicating a missing predicate (e.g., lock-known for a locked drawer).

        # We'll record which drawers/handles are locked or unlocked by attempting to pull them.
        print("[Exploration] Checking drawer lock status via handle pulls...")
        for i, (handle, drawer) in enumerate(zip(handle_list, drawer_list)):
            try:
                # Move to handle location (assume all handles are on 'floor' at start)
                obs, reward, done = execute_go(
                    env, task, from_location='ready-pose', to_location='floor'
                )
                # Pick up the handle
                obs, reward, done = execute_pick(
                    env, task, object=handle, location='floor'
                )
                # Try to pull the drawer using the handle
                obs, reward, done = execute_pull(
                    env, task, drawer=drawer, handle=handle, location='floor'
                )
                print(f"[Exploration] Pulled {drawer} with {handle}: SUCCESS (unlocked)")
            except Exception as e:
                print(f"[Exploration] Pulled {drawer} with {handle}: FAILED (possibly locked) - {e}")
                # This likely means the drawer is locked and the missing predicate is 'lock-known'
                # or the drawer-locked predicate is not satisfied.
                continue
            finally:
                # If holding the handle, try to place it back (if possible)
                try:
                    obs, reward, done = execute_place(
                        env, task, object=handle, drawer=drawer, location='floor'
                    )
                except Exception:
                    pass

        print("[Exploration] Exploration phase complete. If any drawer could not be pulled, it is likely locked and the missing predicate is related to lock status (e.g., lock-known).")

        # === Main Task Plan ===
        # Example: Place dice1 and dice2 into available drawers (unlocked ones), then return to ready-pose

        # For each dice, try to place it in an available (unlocked and empty) drawer
        for dice in dice_list:
            placed = False
            for i, (handle, drawer) in enumerate(zip(handle_list, drawer_list)):
                try:
                    # Move to dice location
                    obs, reward, done = execute_go(
                        env, task, from_location='ready-pose', to_location='floor'
                    )
                    # Pick up the dice
                    obs, reward, done = execute_pick(
                        env, task, object=dice, location='floor'
                    )
                    # Move to handle location (if not already there)
                    obs, reward, done = execute_go(
                        env, task, from_location='floor', to_location='floor'
                    )
                    # Pick up the handle
                    obs, reward, done = execute_pick(
                        env, task, object=handle, location='floor'
                    )
                    # Pull the drawer open
                    obs, reward, done = execute_pull(
                        env, task, drawer=drawer, handle=handle, location='floor'
                    )
                    # Place the dice in the drawer
                    obs, reward, done = execute_place(
                        env, task, object=dice, drawer=drawer, location='floor'
                    )
                    print(f"[Task] Placed {dice} in {drawer}.")
                    placed = True
                    break
                except Exception as e:
                    print(f"[Task] Could not place {dice} in {drawer}: {e}")
                    continue
            if not placed:
                print(f"[Task] Could not place {dice} in any drawer.")

        # Return to ready-pose at the end
        try:
            obs, reward, done = execute_go(
                env, task, from_location='floor', to_location='ready-pose'
            )
            print("[Task] Robot returned to ready-pose.")
        except Exception as e:
            print(f"[Task] Could not return to ready-pose: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
