# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Names (from feedback and observation) ===
        object_names = ['dice1', 'dice2']
        handle_names = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_names = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # === Consistency Check: Ensure all declared objects are present ===
        declared_objects = set(object_names + handle_names + drawer_names)
        missing_objects = [obj for obj in declared_objects if obj not in positions]
        if missing_objects:
            print("[Warning] The following declared objects are missing in the environment positions:", missing_objects)

        # === Debug: Print object names and positions for verification ===
        print("[Debug] Object names:", object_names)
        print("[Debug] Handle names:", handle_names)
        print("[Debug] Drawer names:", drawer_names)
        print("[Debug] Positions keys:", list(positions.keys()))

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration knowledge suggest we need to check for missing predicates (e.g., lock-known, identified, etc.)
        # We'll attempt to use available skills to interact with objects and observe any issues.

        # For this example, let's try to pick up each object and see if any predicate is missing.
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to pick up {obj}...")
                # Try to pick the object from the floor at 'floor' location
                obs, reward, done = execute_pick(env, task, obj, 'floor')
                if done:
                    print(f"[Exploration] Successfully picked up {obj}.")
                    # Place it back in the drawer_bottom if possible
                    print(f"[Exploration] Attempting to place {obj} in drawer_bottom...")
                    obs, reward, done = execute_place(env, task, obj, 'drawer_bottom', 'drawer-area')
                    if done:
                        print(f"[Exploration] Successfully placed {obj} in drawer_bottom.")
                else:
                    print(f"[Exploration] Could not pick up {obj}.")
            except Exception as e:
                print(f"[Error] Exception during pick/place of {obj}: {e}")

        # Try to interact with handles and drawers to check for lock-known or other predicates
        for handle, drawer in zip(handle_names, drawer_names):
            try:
                print(f"[Exploration] Attempting to pick handle {handle}...")
                obs, reward, done = execute_pick(env, task, handle, 'floor')
                if done:
                    print(f"[Exploration] Picked handle {handle}. Attempting to pull drawer {drawer}...")
                    obs, reward, done = execute_pull(env, task, drawer, handle, 'drawer-area')
                    if done:
                        print(f"[Exploration] Pulled drawer {drawer} with handle {handle}.")
                    else:
                        print(f"[Exploration] Could not pull drawer {drawer} (may be locked or missing predicate).")
                else:
                    print(f"[Exploration] Could not pick handle {handle}.")
            except Exception as e:
                print(f"[Error] Exception during handle/drawer interaction: {e}")

        # Try to push drawers closed if open
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Attempting to push drawer {drawer} closed...")
                obs, reward, done = execute_push(env, task, drawer, 'drawer-area')
                if done:
                    print(f"[Exploration] Pushed drawer {drawer} closed.")
                else:
                    print(f"[Exploration] Could not push drawer {drawer} (may already be closed or missing predicate).")
            except Exception as e:
                print(f"[Error] Exception during push of {drawer}: {e}")

        # Try to move the robot to ready-pose and back to floor
        try:
            print("[Exploration] Attempting to move robot to ready-pose...")
            obs, reward, done = execute_go(env, task, 'drawer-area', 'ready-pose')
            if done:
                print("[Exploration] Robot moved to ready-pose.")
            else:
                print("[Exploration] Could not move robot to ready-pose.")
        except Exception as e:
            print(f"[Error] Exception during move to ready-pose: {e}")

        try:
            print("[Exploration] Attempting to move robot to floor...")
            obs, reward, done = execute_go(env, task, 'ready-pose', 'floor')
            if done:
                print("[Exploration] Robot moved to floor.")
            else:
                print("[Exploration] Could not move robot to floor.")
        except Exception as e:
            print(f"[Error] Exception during move to floor: {e}")

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
