# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Names (from feedback and sample problem) ===
        dice_name = 'dice1'
        drawer_name = 'drawer_bottom'
        handle_name = 'handle_bottom'
        ready_pose = 'ready-pose'
        floor = 'floor'
        drawer_area = 'drawer-area'

        # === Check for object existence ===
        required_objects = [dice_name, drawer_name, handle_name, ready_pose, floor, drawer_area]
        for obj in required_objects:
            if obj not in positions:
                print(f"[Error] Required object '{obj}' not found in environment positions.")
                return

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we may need to check for a missing predicate (e.g., is the drawer open/unlocked?)
        # We'll attempt to open the drawer, and if it fails, we can infer the missing predicate (e.g., drawer-locked).
        # For this code, we will attempt the plan and handle errors gracefully.

        # === Oracle Plan Execution ===
        # 1. Move to dice location (floor)
        try:
            print(f"[Task] Moving to dice location: {floor}")
            obs, reward, done, info = execute_go(env, task, from_location=ready_pose, to_location=floor)
        except Exception as e:
            print(f"[Error] Failed to move to dice location: {e}")
            return

        # 2. Pick up the dice
        try:
            print(f"[Task] Picking up dice: {dice_name}")
            obs, reward, done, info = execute_pick(env, task, object_name=dice_name, location=floor)
        except Exception as e:
            print(f"[Error] Failed to pick up dice: {e}")
            return

        # 3. Return to ready pose (if required by need-ready flag)
        try:
            print(f"[Task] Returning to ready pose")
            obs, reward, done, info = execute_go_ready(env, task, from_location=floor)
        except Exception as e:
            print(f"[Warning] Could not return to ready pose: {e}")

        # 4. Move to drawer area
        try:
            print(f"[Task] Moving to drawer area: {drawer_area}")
            obs, reward, done, info = execute_go(env, task, from_location=ready_pose, to_location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to move to drawer area: {e}")
            return

        # 5. Pick up the handle (required for pulling the drawer)
        try:
            print(f"[Task] Picking up handle: {handle_name}")
            obs, reward, done, info = execute_pick_handle(env, task, handle_name=handle_name, location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to pick up handle: {e}")
            return

        # 6. Pull the drawer open
        try:
            print(f"[Task] Pulling open the drawer: {drawer_name}")
            obs, reward, done, info = execute_pull(env, task, drawer_name=drawer_name, handle_name=handle_name, location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to pull open the drawer: {e}")
            print("[Exploration] Drawer may be locked or missing a required predicate (e.g., drawer-unlocked).")
            return

        # 7. Place the dice in the drawer
        try:
            print(f"[Task] Placing dice '{dice_name}' in drawer '{drawer_name}'")
            obs, reward, done, info = execute_place(env, task, object_name=dice_name, drawer_name=drawer_name, location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to place dice in drawer: {e}")
            return

        # 8. Return to ready pose (if required)
        try:
            print(f"[Task] Returning to ready pose")
            obs, reward, done, info = execute_go_ready(env, task, from_location=drawer_area)
        except Exception as e:
            print(f"[Warning] Could not return to ready pose: {e}")

        print("[Task] Successfully put the dice into the drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()

# Notes:
# - All skill invocations use only the predefined skills from skill_code.
# - The code checks for object existence and handles errors gracefully.
# - The code attempts the full plan, and if an action fails (e.g., due to a missing predicate like drawer-locked), it prints an exploration message.
# - No new skill functions are defined; only the provided ones are used.
# - The code is ready to be executed in the RLBench simulation environment.