# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from PDDL and feedback) ===
        dice_name = 'dice1'
        drawer_name = 'drawer_bottom'
        handle_name = 'handle_bottom'
        ready_pose = 'ready-pose'
        floor = 'floor'
        drawer_area = 'drawer-area'

        # Defensive: Check that all required objects are present
        for key in [dice_name, drawer_name, handle_name, ready_pose, floor, drawer_area]:
            if key not in positions:
                print(f"[Error] Required object or location '{key}' not found in positions dictionary.")
                return

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may need to check for a missing predicate,
        # such as whether the drawer is unlocked or if the handle is present.
        # We'll attempt to interact with the handle and drawer, and if an action fails, we log the missing predicate.

        # 1. Move to the dice location (assume dice is on the floor)
        try:
            print(f"[Task] Moving to dice location: {floor}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=floor,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to move to dice location: {e}")
            return

        # 2. Pick up the dice
        try:
            print(f"[Task] Picking up dice: {dice_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=dice_name,
                location=floor,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to pick up dice: {e}")
            return

        # 3. Return to ready-pose (if required by need-ready flag)
        try:
            print(f"[Task] Returning to ready-pose after pick (if required)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=floor,
                max_steps=100
            )
        except Exception as e:
            print(f"[Warning] execute_go_ready may not be required or failed: {e}")

        # 4. Move to the handle location (assume handle is on the floor near the drawer)
        try:
            print(f"[Task] Moving to handle location: {floor}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=floor,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to move to handle location: {e}")
            return

        # 5. Pick up the handle (for pulling the drawer open)
        try:
            print(f"[Task] Picking up handle: {handle_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=floor,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to pick up handle: {e}")
            return

        # 6. Pull the drawer open using the handle
        try:
            print(f"[Task] Pulling open the drawer: {drawer_name} with handle: {handle_name}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=floor,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to pull open the drawer: {e}")
            print("[Exploration] Possible missing predicate: drawer may be locked or handle not attached.")
            return

        # 7. Move to ready-pose after pull (if required)
        try:
            print(f"[Task] Returning to ready-pose after pull (if required)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=floor,
                max_steps=100
            )
        except Exception as e:
            print(f"[Warning] execute_go_ready may not be required or failed: {e}")

        # 8. Move to the drawer area to place the dice
        try:
            print(f"[Task] Moving to drawer area: {drawer_area}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=drawer_area,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to move to drawer area: {e}")
            return

        # 9. Place the dice into the drawer
        try:
            print(f"[Task] Placing dice: {dice_name} into drawer: {drawer_name}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=dice_name,
                drawer_name=drawer_name,
                location=drawer_area,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to place dice into drawer: {e}")
            print("[Exploration] Possible missing predicate: drawer may not be open or is full.")
            return

        # 10. Return to ready-pose after place (if required)
        try:
            print(f"[Task] Returning to ready-pose after place (if required)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_area,
                max_steps=100
            )
        except Exception as e:
            print(f"[Warning] execute_go_ready may not be required or failed: {e}")

        # 11. (Optional) Push the drawer closed
        try:
            print(f"[Task] Pushing the drawer closed: {drawer_name}")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_area,
                max_steps=100
            )
        except Exception as e:
            print(f"[Warning] Could not push drawer closed: {e}")

        print("[Task] Successfully completed: Put dice into drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
