# run_skeleton_task.py (Completed with Exploration Phase and Error Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Object List Consistency Check ===
        # Ensure all objects in positions are unique and valid
        object_names = list(positions.keys())
        if len(object_names) != len(set(object_names)):
            print("[Warning] Duplicate object names found in object_positions!")
            object_names = list(set(object_names))
        if not object_names:
            print("[Error] No objects found in environment!")
            return

        # === Location List Extraction ===
        # Try to extract all unique locations from positions
        location_names = set()
        for obj in object_names:
            pos = positions[obj]
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
            elif isinstance(pos, (tuple, list)) and len(pos) == 3:
                # If positions are just coordinates, assign a dummy location name
                location_names.add('loc_' + obj)
            else:
                location_names.add('unknown')
        location_names = list(location_names)
        if not location_names:
            print("[Error] No locations found in environment!")
            return

        # === Robot Initial Location Guess ===
        # Try to get robot's initial location from observation or default to first location
        robot_location = None
        if hasattr(task, 'get_robot_location'):
            try:
                robot_location = task.get_robot_location()
            except Exception:
                robot_location = location_names[0]
        else:
            robot_location = location_names[0]

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to explore the environment to discover which predicate is missing (e.g., lock-known, temperature-known, etc.)
        # We'll use the available skills to perform exploration actions.
        # For this, we use the available skill names:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # We'll attempt to use 'execute_pull' on objects to see if a lock-known predicate is missing.

        print("[Exploration] Starting exploration phase to identify missing predicate...")

        exploration_success = False
        for obj in object_names:
            try:
                # Move to the object's location if not already there
                obj_pos = positions[obj]
                if isinstance(obj_pos, dict) and 'location' in obj_pos:
                    obj_loc = obj_pos['location']
                else:
                    obj_loc = 'loc_' + obj

                if robot_location != obj_loc:
                    try:
                        print(f"[Exploration] Moving robot from {robot_location} to {obj_loc} for object {obj}...")
                        obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=obj_loc)
                        robot_location = obj_loc
                    except Exception as e:
                        print(f"[Error] Failed to move robot to {obj_loc} for object {obj}: {e}")
                        continue

                # Try to pick the object (if possible)
                try:
                    print(f"[Exploration] Attempting to pick object {obj} at {obj_loc}...")
                    obs, reward, done = execute_pick(env, task, object_name=obj, location=obj_loc)
                except Exception as e:
                    print(f"[Warning] Could not pick object {obj}: {e}")

                # Try to pull the object (to check for lock-known predicate)
                try:
                    print(f"[Exploration] Attempting to pull object {obj} at {obj_loc}...")
                    obs, reward, done = execute_pull(env, task, object_name=obj, location=obj_loc)
                    print(f"[Exploration] Pull action succeeded for {obj}. Likely missing predicate: lock-known.")
                    exploration_success = True
                    break  # Found the missing predicate, can stop exploration
                except Exception as e:
                    print(f"[Info] Pull action failed for {obj}: {e}")
                    # Try other exploration actions if needed

                # Try to sweep or rotate as further exploration (if available)
                try:
                    print(f"[Exploration] Attempting to sweep object {obj} at {obj_loc}...")
                    obs, reward, done = execute_sweep(env, task, object_name=obj, location=obj_loc)
                except Exception as e:
                    print(f"[Info] Sweep action failed for {obj}: {e}")

            except Exception as e:
                print(f"[Error] Exception during exploration for object {obj}: {e}")

        if not exploration_success:
            print("[Exploration] Could not identify missing predicate via pull. Trying other predicates...")

            # Try to use other skills to identify temperature-known, durability-known, etc.
            for obj in object_names:
                try:
                    obj_pos = positions[obj]
                    if isinstance(obj_pos, dict) and 'location' in obj_pos:
                        obj_loc = obj_pos['location']
                    else:
                        obj_loc = 'loc_' + obj

                    if robot_location != obj_loc:
                        try:
                            print(f"[Exploration] Moving robot from {robot_location} to {obj_loc} for object {obj}...")
                            obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=obj_loc)
                            robot_location = obj_loc
                        except Exception as e:
                            print(f"[Error] Failed to move robot to {obj_loc} for object {obj}: {e}")
                            continue

                    # Try to pick the object (again) to see if weight-known or durability-known is missing
                    try:
                        print(f"[Exploration] Attempting to pick object {obj} at {obj_loc} for weight/durability...")
                        obs, reward, done = execute_pick(env, task, object_name=obj, location=obj_loc)
                        print(f"[Exploration] Pick action succeeded for {obj}. Possibly missing predicate: weight-known or durability-known.")
                        exploration_success = True
                        break
                    except Exception as e:
                        print(f"[Info] Pick action failed for {obj}: {e}")

                except Exception as e:
                    print(f"[Error] Exception during secondary exploration for object {obj}: {e}")

        if exploration_success:
            print("[Exploration] Exploration phase completed. Missing predicate likely identified.")
        else:
            print("[Exploration] Exploration phase completed. Could not identify missing predicate.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # Here you would execute the oracle plan using the available skills, now that you have explored the environment.
        # For this template, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()