# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from feedback and PDDL) ===
        # Objects: dice1, dice2, drawer_top, drawer_middle, drawer_bottom
        # Handles: drawer_top, drawer_middle, drawer_bottom (are handles themselves)
        # Locations: floor, drawer-area, ready-pose

        # For this example, we will insert dice1 into drawer_top
        die_name = 'dice1'
        drawer_name = 'drawer_top'
        handle_name = 'drawer_top'  # handle is the same as drawer in this domain
        ready_pose = 'ready-pose'
        drawer_area = 'drawer-area'
        floor = 'floor'

        # Safety: Check that all required objects are present
        required_objects = [die_name, drawer_name, handle_name]
        for obj in required_objects:
            if obj not in positions:
                print(f"[Error] Required object '{obj}' not found in object positions.")
                return

        # === Exploration Phase: Find Missing Predicate (lock-known) ===
        # The feedback and exploration domain suggest that the robot may need to discover the lock state of the drawer by pulling the handle.
        # We simulate this by attempting to pull the handle and observing if the predicate 'lock-known' is needed.

        print("[Exploration] Attempting to pull the drawer handle to discover lock state...")

        try:
            # Move to ready pose if not already there
            obs, reward, done = execute_go(env, task, from_location=positions[ready_pose], to_location=positions[drawer_area])
            if done:
                print("[Exploration] Task ended unexpectedly during move to drawer area.")
                return

            # Pick the handle (drawer handle is an object on the floor)
            obs, reward, done = execute_pick_handle(env, task, handle_name, positions[drawer_area])
            if done:
                print("[Exploration] Task ended unexpectedly during handle pick.")
                return

            # Try to pull the drawer (this should set lock-known if not already known)
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, positions[drawer_area])
            if done:
                print("[Exploration] Task ended unexpectedly during drawer pull.")
                return

            print("[Exploration] Drawer pull executed. If lock-known was missing, it should now be discovered.")

        except Exception as e:
            print(f"[Exploration] Exception during exploration phase: {e}")
            return

        # === Main Task Plan: Insert die into drawer ===

        try:
            # 1. Move to die location (assume die is on the floor)
            print(f"[Task] Moving to die '{die_name}' at floor...")
            obs, reward, done = execute_go(env, task, from_location=positions[drawer_area], to_location=positions[floor])
            if done:
                print("[Task] Task ended unexpectedly during move to die.")
                return

            # 2. Pick the die
            print(f"[Task] Picking up die '{die_name}'...")
            obs, reward, done = execute_pick_object(env, task, die_name, positions[floor])
            if done:
                print("[Task] Task ended unexpectedly during die pick.")
                return

            # 3. Return to ready pose (required after pick)
            print("[Task] Returning to ready pose after pick...")
            obs, reward, done = execute_go_ready(env, task, positions[floor])
            if done:
                print("[Task] Task ended unexpectedly during go ready after pick.")
                return

            # 4. Move to drawer area
            print(f"[Task] Moving to drawer area to place die in '{drawer_name}'...")
            obs, reward, done = execute_go(env, task, from_location=positions[ready_pose], to_location=positions[drawer_area])
            if done:
                print("[Task] Task ended unexpectedly during move to drawer area.")
                return

            # 5. Place the die into the drawer
            print(f"[Task] Placing die '{die_name}' into drawer '{drawer_name}'...")
            obs, reward, done = execute_place_object(env, task, die_name, drawer_name, positions[drawer_area])
            if done:
                print("[Task] Task ended unexpectedly during die place.")
                return

            # 6. Return to ready pose (required after place)
            print("[Task] Returning to ready pose after place...")
            obs, reward, done = execute_go_ready(env, task, positions[drawer_area])
            if done:
                print("[Task] Task ended unexpectedly during go ready after place.")
                return

            # 7. Push the drawer closed
            print(f"[Task] Pushing drawer '{drawer_name}' closed...")
            obs, reward, done = execute_push(env, task, drawer_name, positions[drawer_area])
            if done:
                print("[Task] Task ended unexpectedly during drawer push.")
                return

            print("[Task] Die successfully inserted into drawer and drawer closed.")

        except Exception as e:
            print(f"[Task] Exception during main task execution: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()