# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Retrieve Object List for Existence Check ===
        # Assume get_object_positions() returns a dict of object_name: position
        object_list = set(positions.keys())

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to check if objects (e.g., 'dice1') exist before acting.
        # We also need to ensure that, if the object is inside a drawer, we calibrate force (handled by skill).
        # Here, we perform an exploration step to identify all objects at their locations.

        # For demonstration, let's try to identify all objects at all locations
        # (In a real plan, this would be more targeted.)
        # We'll use the available skill 'execute_go' to move, and 'execute_sweep' or similar to explore if needed.

        # Let's print all objects for debug
        print("[Exploration] Objects in environment:", object_list)

        # Example: Suppose we want to manipulate 'dice1'
        target_object = 'dice1'
        if target_object not in object_list:
            print(f"[Error] Target object '{target_object}' not found in environment. Aborting task.")
            return

        # Find the position/location of the target object
        target_pos = positions[target_object]
        print(f"[Exploration] '{target_object}' is at position {target_pos}")

        # Suppose we need to move the robot to the object's location before picking
        # Let's find the robot's current location
        robot_location = None
        for name, pos in positions.items():
            if 'robot' in name:
                robot_location = pos
                break
        if robot_location is None:
            print("[Error] Robot location not found in positions. Aborting task.")
            return

        # === Move to the object's location ===
        try:
            print(f"[Action] Moving robot to '{target_object}' at {target_pos}")
            obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=target_pos)
            if done:
                print("[Task] Task ended after move!")
                return
        except Exception as e:
            print(f"[Exception] Error during execute_go: {e}")
            return

        # === Exploration: Identify the object (simulate missing predicate discovery) ===
        # In a real exploration, we might use a skill like execute_sweep or execute_go_identify
        # Here, we use execute_sweep if available to simulate exploration
        try:
            print(f"[Exploration] Identifying '{target_object}' at {target_pos}")
            obs, reward, done = execute_sweep(env, task, obj=target_object, p=target_pos)
            if done:
                print("[Task] Task ended after exploration!")
                return
        except Exception as e:
            print(f"[Exception] Error during execute_sweep: {e}")
            return

        # === Pick the object ===
        try:
            print(f"[Action] Picking up '{target_object}' at {target_pos}")
            obs, reward, done = execute_pick(env, task, o=target_object, p=target_pos)
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Exception] Error during execute_pick: {e}")
            return

        # === Place the object (example: into a drawer if required) ===
        # For demonstration, let's assume we want to place it into 'drawer_middle'
        drawer_name = 'drawer_middle'
        if drawer_name not in object_list:
            print(f"[Error] Drawer '{drawer_name}' not found in environment. Skipping place action.")
        else:
            drawer_pos = positions[drawer_name]
            try:
                print(f"[Action] Placing '{target_object}' into '{drawer_name}' at {drawer_pos}")
                obs, reward, done = execute_place(env, task, o=target_object, d=drawer_name, p=drawer_pos)
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Exception] Error during execute_place: {e}")

        # === Push the drawer closed (if required) ===
        try:
            print(f"[Action] Pushing '{drawer_name}' closed at {drawer_pos}")
            obs, reward, done = execute_push(env, task, d=drawer_name, p=drawer_pos)
            if done:
                print("[Task] Task ended after push!")
                return
        except Exception as e:
            print(f"[Exception] Error during execute_push: {e}")

        # === Exploration: Check for missing predicates (simulate) ===
        # If the plan fails, we could try to discover missing predicates by further exploration
        # For example, try to pull the drawer handle to check lock status
        handle_name = 'drawer_middle_handle'
        if handle_name in object_list:
            handle_pos = positions[handle_name]
            try:
                print(f"[Exploration] Pulling handle '{handle_name}' at {handle_pos} to check lock status")
                obs, reward, done = execute_pull(env, task, d=drawer_name, h=handle_name, p=handle_pos)
                if done:
                    print("[Task] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Exception] Error during execute_pull: {e}")

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
