# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # EXPLORATION PHASE: Try existing skill predicates to identify missing predicate
        
        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]
        
        # Define a sequence of exploration actions that exercise most skill preconditions
        # The idea is to probe the environment using available skills and see which predicates are missing

        # Attempt each available skill in a safe, dummy fashion, catching exceptions and reporting them
        # This will surface which predicates or preconditions are missing in problem/observation
        exploration_report = []
        print("[Exploration] Executing available skills to detect missing predicate(s):")
        for skill_name in available_skills:
            try:
                # Choose arguments heuristically based on assumptions about the environment/object names.
                # Retrieve possible object/location keys
                obj_keys = [k for k in positions.keys() if not k.startswith('loc') and not k == 'robot']
                loc_keys = [k for k in positions.keys() if k.startswith('loc') or 'drawer' in k or 'table' in k]
                # Some environments may have 'ready_pose', 'table', or 'drawer' as locations
                if 'ready_pose' not in loc_keys:
                    loc_keys.append('ready_pose')
                # Safely get first available objects/locations
                obj = obj_keys[0] if obj_keys else None
                loc = loc_keys[0] if loc_keys else None
                loc2 = loc_keys[1] if len(loc_keys) > 1 else loc
                # For drawer-specific actions (by name), try to find drawer-related keys
                drawer = None
                for k in positions:
                    if 'drawer' in k:
                        drawer = k
                        break
                # For handle-related actions, try to find handle object
                handle_obj = None
                for k in obj_keys:
                    if 'handle' in k:
                        handle_obj = k
                        break
                # For execute_pick: try with generic object and location
                if skill_name == 'execute_pick':
                    if obj and loc:
                        print(f"[Exploration] Trying {skill_name}({obj}, {loc}) ...")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                elif skill_name == 'execute_place':
                    # Requires object, drawer, location
                    if obj and drawer and loc:
                        print(f"[Exploration] Trying {skill_name}({obj}, {drawer}, {loc}) ...")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                elif skill_name == 'execute_push':
                    # Requires drawer, location
                    if drawer and loc:
                        print(f"[Exploration] Trying {skill_name}({drawer}, {loc}) ...")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                elif skill_name == 'execute_pull':
                    # Requires drawer, handle object, location
                    if drawer and handle_obj and loc:
                        print(f"[Exploration] Trying {skill_name}({drawer}, {handle_obj}, {loc}) ...")
                        obs, reward, done = execute_pull(env, task, drawer, handle_obj, loc)
                elif skill_name == 'execute_sweep':
                    # Try with object and location
                    if obj and loc:
                        print(f"[Exploration] Trying {skill_name}({obj}, {loc}) ...")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                elif skill_name == 'execute_rotate':
                    # Try with object, location if possible (dummy invocation)
                    if obj and loc:
                        print(f"[Exploration] Trying {skill_name}({obj}, {loc}) ...")
                        obs, reward, done = execute_rotate(env, task, obj, loc)
                elif skill_name == 'execute_go':
                    # Try moving between two locations
                    if loc and loc2 and loc != loc2:
                        print(f"[Exploration] Trying {skill_name}({loc}, {loc2}) ...")
                        obs, reward, done = execute_go(env, task, loc, loc2)
                elif skill_name == 'execute_gripper':
                    # execute_gripper usually takes no arguments
                    print(f"[Exploration] Trying {skill_name}() ...")
                    obs, reward, done = execute_gripper(env, task)
                else:
                    print(f"[Exploration] Skill {skill_name} not recognized for auto-exploration.")
                    
                exploration_report.append((skill_name, True, "No error"))
            except Exception as e:
                print(f"[Exploration] {skill_name}: Exception occurred: {e}")
                exploration_report.append((skill_name, False, str(e)))
        
        # After exploration, print out which actions failed due to missing predicates, if any
        print("\n[Exploration] Summary Report:")
        for (skill, success, msg) in exploration_report:
            if not success:
                print(f"  Skill '{skill}' failed with message: {msg}")
            else:
                print(f"  Skill '{skill}' succeeded.")

        print("\n[Exploration] End of predicate probing phase.\n")
        # End of the exploration phase

        # After identifying missing predicates using exploration above, 
        # the user should fill-in/fix the PDDL problem/domain to supply these predicates.
        # Normally, at this point, the real plan for the oracle task would be run here.
        #
        # Placeholder:
        print("[Task] -- Insert the actual oracle plan execution logic here after PDDL is fixed --")
        print("[Task] The code above has attempted all available skills to probe for missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
