# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates we must verify dice location and orientation before storing.
        # We'll use available skills to explore and check the environment.

        # Define object and location names based on the observation and domain
        dice_names = ['dice1', 'dice2']
        drawer_names = ['drawer-bottom', 'drawer-middle', 'drawer-top']
        handle_names = ['handle-bottom', 'handle-middle', 'handle-top']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # For this example, we focus on dice1 and drawer-bottom
        dice = 'dice1'
        drawer = 'drawer-bottom'
        handle = 'handle-bottom'
        floor = 'floor'
        ready_pose = 'ready-pose'
        drawer_area = 'drawer-area'

        # 1. Move robot to the dice location (assume dice is on the floor)
        try:
            print(f"[Exploration] Moving robot to {floor} to check {dice} location.")
            obs, reward, done = execute_go(env, task, from_location=ready_pose, to_location=floor)
        except Exception as e:
            print(f"[Error] Failed to move to {floor}: {e}")

        # 2. (Optional) Check dice orientation if such a skill exists (not in available skills)
        # Since we don't have an explicit orientation check, we proceed.

        # 3. Pick up the dice from the floor
        try:
            print(f"[Exploration] Picking up {dice} from {floor}.")
            obs, reward, done = execute_pick(env, task, object_name=dice, location=floor)
        except Exception as e:
            print(f"[Error] Failed to pick {dice}: {e}")

        # 4. Move to ready-pose (required by domain after pick)
        try:
            print(f"[Exploration] Moving to ready-pose after pick.")
            obs, reward, done = execute_go_ready(env, task, from_location=floor)
        except Exception as e:
            print(f"[Error] Failed to go to ready-pose: {e}")

        # 5. Move to drawer area
        try:
            print(f"[Exploration] Moving to {drawer_area} to access {drawer}.")
            obs, reward, done = execute_go(env, task, from_location=ready_pose, to_location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to move to {drawer_area}: {e}")

        # 6. Pick up the handle for the drawer
        try:
            print(f"[Exploration] Picking up handle {handle} at {drawer_area}.")
            obs, reward, done = execute_pick(env, task, object_name=handle, location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to pick handle {handle}: {e}")

        # 7. Pull the drawer open using the handle
        try:
            print(f"[Exploration] Pulling {drawer} open with {handle}.")
            obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_name=handle, location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to pull {drawer}: {e}")

        # 8. Place the dice into the open drawer
        try:
            print(f"[Exploration] Placing {dice} into {drawer} at {drawer_area}.")
            obs, reward, done = execute_place(env, task, object_name=dice, drawer_name=drawer, location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to place {dice} in {drawer}: {e}")

        # 9. Move to ready-pose after place
        try:
            print(f"[Exploration] Moving to ready-pose after place.")
            obs, reward, done = execute_go_ready(env, task, from_location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to go to ready-pose: {e}")

        # 10. Push the drawer closed
        try:
            print(f"[Exploration] Pushing {drawer} closed at {drawer_area}.")
            obs, reward, done = execute_push(env, task, drawer_name=drawer, location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to push {drawer}: {e}")

        # 11. Move to ready-pose after push
        try:
            print(f"[Exploration] Moving to ready-pose after push.")
            obs, reward, done = execute_go_ready(env, task, from_location=drawer_area)
        except Exception as e:
            print(f"[Error] Failed to go to ready-pose: {e}")

        print("[Exploration] Task completed. If any step failed, check the error messages above.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
