# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()  # Should be a dict of ('object_name': position)

        # === Begin Exploration Phase ===
        # As the feedback suggests a required predicate is missing in the PDDL,
        # We must perform an "exploration" phase using only the available skills to help discover what's missing.
        print("[Exploration] Beginning predicate exploration phase.")

        # Get a list of objects and locations from positions
        object_names = [k for k in positions.keys() if not k.startswith("location_") and not k.startswith("drawer_")]
        location_names = [k for k in positions.keys() if k.startswith("location_")]
        drawer_names = [k for k in positions.keys() if k.startswith("drawer_")]
        
        # Flag to track which predicates we have tried to uncover
        explored_predicates = set()
        
        # Available primitive skill names
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Try to visit all locations using execute_go (might correspond to robot-at/location predicates)
        try:
            curr_location = None
            for loc in location_names:
                if curr_location is None:
                    curr_location = loc
                    continue  # Assume we start at this location
                try:
                    print(f"[Exploration] Moving robot from {curr_location} to {loc}.")
                    obs, reward, done = execute_go(env, task, curr_location, loc)
                    curr_location = loc
                except Exception as e:
                    print(f"[Exploration][Error] execute_go {curr_location} -> {loc}: {str(e)}")
            explored_predicates.add('robot-at')
        except Exception as ex:
            print(f"[Exploration][Error] execute_go phase: {str(ex)}")

        # 2. Try to pick up all objects using execute_pick (try both execute_pick and execute_pull as needed)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc}.")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    explored_predicates.add('holding')
                    print(f"[Exploration] Picked up {obj} at {loc}.")
                    # Try a push or pull if relevant
                    for drawer in drawer_names:
                        try:
                            print(f"[Exploration] Trying to pull drawer {drawer} with {obj} at {loc}.")
                            obs, reward, done = execute_pull(env, task, drawer, obj, loc)
                            explored_predicates.add('lock-known')
                            print(f"[Exploration] Pulled {drawer} with {obj} at {loc}.")
                        except Exception as pull_ex:
                            print(f"[Exploration][Info] Could not pull {drawer} with {obj}: {str(pull_ex)}")
                    # Place it back somewhere
                    for drawer in drawer_names:
                        try:
                            print(f"[Exploration] Placing {obj} in drawer {drawer} at {loc}.")
                            obs, reward, done = execute_place(env, task, obj, drawer, loc)
                            explored_predicates.add('in-drawer')
                            print(f"[Exploration] Placed {obj} in {drawer} at {loc}.")
                            break
                        except Exception as place_ex:
                            print(f"[Exploration][Info] Could not place {obj} in {drawer}: {str(place_ex)}")
                except Exception as pick_ex:
                    print(f"[Exploration][Info] Could not pick {obj} at {loc}: {str(pick_ex)}")

        # 3. Try handle-focused actions (if possible)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting execute_sweep on {obj} at {loc}.")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    explored_predicates.add('swept')
                except Exception as sweep_ex:
                    print(f"[Exploration][Info] Could not sweep {obj} at {loc}: {str(sweep_ex)}")
        
        # 4. Use gripper action if available
        try:
            print("[Exploration] Trying execute_gripper primitive.")
            obs, reward, done = execute_gripper(env, task)
            explored_predicates.add('gripper')
        except Exception as gripper_ex:
            print(f"[Exploration][Info] Could not execute gripper: {str(gripper_ex)}")

        print("[Exploration] Explored predicates:", explored_predicates)
        print("[Exploration] Finished exploration phase. Check simulation video/logs for missing predicate cues.")

        # === (Optional) Here, you would continue to run the actual plan once predicate understanding is complete ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()