# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only existing skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Names ===
        positions = get_object_positions()
        object_names = list(positions.keys()) if isinstance(positions, dict) else []

        # === Exploration Phase to Determine Missing Predicate ===
        print("[Exploration] Starting predicate exploration to diagnose task blocking predicates...")
        exploration_locations = []
        for obj in object_names:
            # Try to get location names from positions dictionary values
            try:
                pos_info = positions[obj]
                if isinstance(pos_info, dict) and "location" in pos_info:
                    exploration_locations.append(pos_info["location"])
            except Exception:
                continue
        if not exploration_locations and object_names:
            # Fallback: If we can't extract locations above, use standard location names or mock ones
            exploration_locations = ['location_1', 'location_2']

        # Try to visit every object and perform a "go" and "pick" to check for need-ready predicate blocking
        for obj in object_names:
            try:
                pos_info = positions[obj]
                obj_location = None
                if isinstance(pos_info, dict) and "location" in pos_info:
                    obj_location = pos_info["location"]
                elif isinstance(pos_info, (tuple, list)) and len(pos_info) >= 2:
                    obj_location = pos_info
                else:
                    obj_location = None
                if obj_location is not None:
                    print(f"[Exploration] Attempt execute_go to: {obj_location}")
                    try:
                        obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location=obj_location)
                    except Exception as ex:
                        print(f"[Exploration] execute_go Exception for {obj}: {ex}")

                print(f"[Exploration] Attempt to execute_pick on {obj} at {obj_location}")
                try:
                    obs, reward, done = execute_pick(env, task, object_name=obj, location=obj_location)
                except Exception as ex:
                    print(f"[Exploration] execute_pick Exception for {obj}: {ex}")

                # After pick, try to detect need-ready presence
                need_ready_flag = False
                if hasattr(task, 'get_internal_state'):
                    state = task.get_internal_state()
                    if isinstance(state, dict) and state.get('need-ready', False):
                        need_ready_flag = True
                # Fallback: try by querying obs or env if possible
                if 'need-ready' in str(obs) or 'need_ready' in str(obs):
                    need_ready_flag = True

                if need_ready_flag:
                    print("[Exploration] Detected 'need-ready' predicate after pick action.")

                    print("[Exploration] Attempting to clear 'need-ready' with execute_go_ready.")
                    try:
                        obs, reward, done = execute_go_ready(env, task, from_location=obj_location)
                        print("[Exploration] Cleared 'need-ready' predicate using execute_go_ready.")
                    except Exception as ex:
                        print(f"[Exploration] execute_go_ready Exception: {ex}")

            except Exception as e:
                print(f"[Exploration] Error during exploration on object {obj}: {e}")

        print("[Exploration] Exploration phase complete.")

        # === PLAN EXECUTION (Example Sequence). ===
        # Use feedback to reinforce plan safety
        # Check: object exists before place, verify drawer state before placing
        for obj in object_names:
            try:
                pos_info = positions[obj]
                obj_location = None
                if isinstance(pos_info, dict) and "location" in pos_info:
                    obj_location = pos_info["location"]
                elif isinstance(pos_info, (tuple, list)) and len(pos_info) >= 2:
                    obj_location = pos_info
                else:
                    obj_location = None

                # Example: Suppose we have a drawer called "drawer1"
                drawer_name = "drawer1"
                drawer_open = False

                # Check object presence (Feedback step)
                if obj not in object_names:
                    print(f"[Plan] ERROR: Object {obj} not in environment. Skipping.")
                    continue

                # Move to object location and pick (using skills)
                if obj_location is not None:
                    print(f"[Plan] Moving to {obj_location} to pick {obj}.")
                    try:
                        obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location=obj_location)
                    except Exception as ex:
                        print(f"[Plan] execute_go failed for {obj}: {ex}")
                        continue

                print(f"[Plan] Picking object {obj} at {obj_location}.")
                try:
                    obs, reward, done = execute_pick(env, task, object_name=obj, location=obj_location)
                except Exception as ex:
                    print(f"[Plan] execute_pick failed for {obj}: {ex}")
                    continue

                # Before placing, verify drawer is open, using feedback suggestion
                if hasattr(task, 'is_drawer_open'):
                    try:
                        drawer_open = task.is_drawer_open(drawer_name)
                    except:
                        drawer_open = False
                # Fallback: Use observation
                if not drawer_open and drawer_name in str(obs):
                    if 'open' in str(obs[drawer_name]):
                        drawer_open = True

                if not drawer_open:
                    print(f"[Plan] Drawer {drawer_name} not open. Attempting to open with execute_pull.")
                    try:
                        obs, reward, done = execute_pull(env, task, drawer_name, obj, obj_location)
                        drawer_open = True
                        print(f"[Plan] Drawer {drawer_name} opened.")
                    except Exception as ex:
                        print(f"[Plan] execute_pull failed for {drawer_name}: {ex}")

                # Place object in the drawer if it's now open
                if drawer_open:
                    print(f"[Plan] Attempting to place {obj} in {drawer_name}.")
                    try:
                        obs, reward, done = execute_place(env, task, object_name=obj, drawer_name=drawer_name, location=obj_location)
                    except Exception as ex:
                        print(f"[Plan] execute_place failed for {obj} in {drawer_name}: {ex}")
                else:
                    print(f"[Plan] ERROR: Drawer {drawer_name} still not open. Aborting place for {obj}.")

                # After placing, check if need-ready is set and clear it
                need_ready = False
                if hasattr(task, 'get_internal_state'):
                    state = task.get_internal_state()
                    if isinstance(state, dict) and state.get('need-ready', False):
                        need_ready = True
                if need_ready or 'need-ready' in str(obs):
                    print(f"[Plan] need-ready flagged after place. Executing execute_go_ready to reset pose.")
                    try:
                        obs, reward, done = execute_go_ready(env, task, from_location=obj_location)
                        print(f"[Plan] Cleared need-ready after place.")
                    except Exception as ex:
                        print(f"[Plan] execute_go_ready failed post-place: {ex}")

            except Exception as ex:
                print(f"[Plan] Exception in object plan for {obj}: {ex}")

        print("[Plan] Plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
