# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we need to check for object existence and drawer state.
        # We'll use exploration skills to identify objects and check drawer state before manipulation.

        # Example object and drawer names (replace with actual names from your environment)
        object_list = [obj for obj in positions if 'object' in obj or 'obj' in obj]
        drawer_list = [obj for obj in positions if 'drawer' in obj]
        handle_list = [obj for obj in positions if 'handle' in obj]

        # For demonstration, pick the first object and drawer found
        if not object_list or not drawer_list:
            print("[Error] No objects or drawers found in the environment positions!")
            return

        target_object = object_list[0]
        target_drawer = drawer_list[0]
        # Try to find a handle for the drawer if available
        target_handle = None
        for h in handle_list:
            if target_drawer in h:
                target_handle = h
                break
        if not target_handle and handle_list:
            target_handle = handle_list[0]

        # Get positions
        object_pos = positions.get(target_object, None)
        drawer_pos = positions.get(target_drawer, None)
        handle_pos = positions.get(target_handle, None) if target_handle else None

        # Safety: Check that all required positions are available
        if object_pos is None or drawer_pos is None:
            print("[Error] Required object or drawer position not found!")
            return

        # === Step 1: Exploration - Identify the object at its location ===
        # Use execute_go_identify to ensure the object is recognized
        try:
            print(f"[Exploration] Identifying object {target_object} at its location.")
            obs, reward, done = execute_go_identify(
                env,
                task,
                from_location='ready-pose',  # Assuming robot starts at ready-pose
                to_location=object_pos,
                object_name=target_object
            )
        except Exception as e:
            print(f"[Exploration Error] Failed to identify object: {e}")
            return

        # === Step 2: Exploration - Check drawer lock state (lock-known) ===
        # Use execute_pull to check lock state if needed
        try:
            print(f"[Exploration] Checking lock state of drawer {target_drawer}.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=target_drawer,
                handle_name=target_handle,
                location=drawer_pos
            )
        except Exception as e:
            print(f"[Exploration Error] Failed to check drawer lock state: {e}")
            return

        # === Step 3: Move to object and pick it up ===
        try:
            print(f"[Task] Moving to object {target_object} at {object_pos}.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=object_pos
            )
            print(f"[Task] Picking up object {target_object}.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=target_object,
                location=object_pos
            )
        except Exception as e:
            print(f"[Task Error] Failed to move and pick object: {e}")
            return

        # === Step 4: Move to drawer and open it (if not already open) ===
        try:
            print(f"[Task] Moving to drawer {target_drawer} at {drawer_pos}.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=object_pos,
                to_location=drawer_pos
            )
            # Open the drawer using the handle if required
            if target_handle:
                print(f"[Task] Pulling handle {target_handle} to open drawer {target_drawer}.")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=target_drawer,
                    handle_name=target_handle,
                    location=drawer_pos
                )
        except Exception as e:
            print(f"[Task Error] Failed to move and open drawer: {e}")
            return

        # === Step 5: Safety Check - Is object still held and drawer open? ===
        # (Assume skill_code or env provides state-checking, or skip if not available)

        # === Step 6: Place object into drawer ===
        try:
            print(f"[Task] Placing object {target_object} into drawer {target_drawer}.")
            # Check if object is still in object_list before placing
            if target_object not in object_list:
                print(f"[Safety] Object {target_object} not in object list, aborting place.")
                return
            obs, reward, done = execute_place(
                env,
                task,
                object_name=target_object,
                drawer_name=target_drawer,
                location=drawer_pos
            )
        except Exception as e:
            print(f"[Task Error] Failed to place object in drawer: {e}")
            return

        # === Step 7: Push drawer closed ===
        try:
            print(f"[Task] Pushing drawer {target_drawer} closed.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=target_drawer,
                location=drawer_pos
            )
        except Exception as e:
            print(f"[Task Error] Failed to push drawer closed: {e}")
            return

        # === Step 8: Return to ready pose ===
        try:
            print(f"[Task] Returning to ready-pose.")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_pos
            )
        except Exception as e:
            print(f"[Task Error] Failed to return to ready-pose: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
