# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest we need to check for missing predicates
        # such as weight-known, durability-known, lock-known, etc.
        # We will use the available skills to explore the environment and try to discover
        # which predicate is missing by attempting to execute actions that would reveal them.

        # For this, we will iterate over all objects and attempt to pick, pull, or otherwise
        # interact with them using the available skills.

        # Get the list of objects and locations from the positions dictionary
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # For demonstration, we will attempt to pick each object and then pull if possible,
        # to see if any predicate (e.g., lock-known) is missing or required.

        for obj_name in object_names:
            obj_pos = positions[obj_name]
            # Try to move to the object's location
            # We assume each object has an associated location, or we use a default location
            # For this example, we use the first location in the list
            if location_names:
                target_location = location_names[0]
                target_location_pos = positions[target_location]
            else:
                # If no location is defined, use the object's own position
                target_location = obj_name
                target_location_pos = obj_pos

            print(f"[Exploration] Moving to location of {obj_name} ({target_location}) at {target_location_pos}")
            try:
                # Use execute_go to move the robot to the target location
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=None,  # Let the skill handle current location
                    to_location=target_location_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Failed to move to {target_location}: {e}")
                continue

            # Try to pick the object (if possible)
            print(f"[Exploration] Attempting to pick {obj_name}")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj_name,
                    location=target_location_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj_name}: {e}")
                continue

            # Try to pull the object (if possible)
            print(f"[Exploration] Attempting to pull {obj_name}")
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    object_name=obj_name,
                    location=target_location_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Failed to pull {obj_name}: {e}")
                continue

            # After each action, check for any new information or missing predicates
            # (In a real implementation, you would inspect the environment state or logs)

        print("[Exploration] Exploration phase completed. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
