# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # The goal is to drop one dice into any open drawer, but feedback indicates
        # that the code must check for drawer openness, object type, and safety.
        # We will explore the environment to determine which predicate is missing
        # (e.g., is the drawer open? is the object a dice? etc.)

        # --- Step 1: Identify all dice and drawers ---
        dice_names = [name for name in positions if 'dice' in name]
        drawer_names = [name for name in positions if 'drawer' in name and 'area' not in name]
        handle_names = [name for name in positions if 'handle' in name]

        # --- Step 2: Try to find an open drawer ---
        open_drawer = None
        for drawer in drawer_names:
            try:
                # Use a predefined skill to check if the drawer is open
                # (Assume skill_code provides a way to check state, or use env/task API)
                # If not, we attempt to open it and see if it succeeds
                # For exploration, try to push (close) and then pull (open) the drawer
                # to see if the state changes as expected.
                # Here, we try to pull the drawer using its handle.
                # First, find the handle for this drawer
                handle = None
                for h in handle_names:
                    if drawer in h:
                        handle = h
                        break
                if handle is None:
                    continue
                # Move to the handle position
                obs, reward, done = execute_go(
                    env, task,
                    from_location='ready-pose',
                    to_location=positions[handle],
                    max_steps=100
                )
                # Pick the handle
                obs, reward, done = execute_pick(
                    env, task,
                    object_name=handle,
                    location=positions[handle],
                    max_steps=100
                )
                # Try to pull the drawer
                obs, reward, done = execute_pull(
                    env, task,
                    drawer_name=drawer,
                    handle_name=handle,
                    location=positions[handle],
                    max_steps=100
                )
                # If pull succeeds, the drawer is now open
                open_drawer = drawer
                print(f"[Exploration] Found openable drawer: {drawer}")
                break
            except Exception as e:
                print(f"[Exploration] Could not open drawer {drawer}: {e}")
                continue

        if open_drawer is None:
            print("[Exploration] No openable drawer found. Exploration failed.")
            return

        # --- Step 3: Find a dice on the floor ---
        dice_to_pick = None
        for dice in dice_names:
            # Check if the dice is on the floor (assume positions or task API can tell)
            # For now, just pick the first dice
            dice_to_pick = dice
            break

        if dice_to_pick is None:
            print("[Exploration] No dice found to pick.")
            return

        # --- Step 4: Move to the dice and pick it ---
        try:
            obs, reward, done = execute_go(
                env, task,
                from_location='ready-pose',
                to_location=positions[dice_to_pick],
                max_steps=100
            )
            obs, reward, done = execute_pick(
                env, task,
                object_name=dice_to_pick,
                location=positions[dice_to_pick],
                max_steps=100
            )
            print(f"[Exploration] Picked up dice: {dice_to_pick}")
        except Exception as e:
            print(f"[Exploration] Failed to pick up dice {dice_to_pick}: {e}")
            return

        # --- Step 5: Move to the open drawer and place the dice inside ---
        try:
            obs, reward, done = execute_go(
                env, task,
                from_location=positions[dice_to_pick],
                to_location=positions[open_drawer],
                max_steps=100
            )
            obs, reward, done = execute_place(
                env, task,
                object_name=dice_to_pick,
                drawer_name=open_drawer,
                location=positions[open_drawer],
                max_steps=100
            )
            print(f"[Exploration] Placed dice {dice_to_pick} into drawer {open_drawer}")
        except Exception as e:
            print(f"[Exploration] Failed to place dice {dice_to_pick} into drawer {open_drawer}: {e}")
            return

        # --- Step 6: Return to ready pose ---
        try:
            obs, reward, done = execute_go(
                env, task,
                from_location=positions[open_drawer],
                to_location=positions['ready-pose'],
                max_steps=100
            )
            print("[Exploration] Returned to ready pose.")
        except Exception as e:
            print(f"[Exploration] Failed to return to ready pose: {e}")

        # --- Step 7: Exploration complete, check for missing predicates ---
        # Based on the feedback, if any of the above steps failed due to a missing predicate
        # (e.g., drawer not open, object not a dice, etc.), we can infer which predicate is missing.
        # For example, if the place action failed because the drawer was not open,
        # then the missing predicate is likely (drawer-open ?d).
        # If the pick action failed because the object was not a dice, then the missing predicate is (is-dice ?o).
        # Log the results for further analysis.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
