# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and drawer names as per the PDDL problem ===
        object_list = ["dice1", "dice2"]
        handle_list = ["handle_bottom", "handle_middle", "handle_top"]
        drawer_list = ["drawer_bottom", "drawer_middle", "drawer_top"]
        location_list = ["floor", "drawer-area", "ready-pose"]

        # === Exploration Phase: Identify missing predicate ===
        # Based on feedback, we need to check if the object is in the object list before proceeding.
        # Also, we need to ensure the drawer choice is valid.

        # For demonstration, let's try to pick and place both dice into a user-chosen drawer.
        # We'll ask the user for the drawer choice, but also check if the object is present.

        # --- Ask user for drawer choice ---
        valid_drawers = ["drawer_top", "drawer_middle", "drawer_bottom"]
        drawer_choice = None
        while True:
            print("Available drawers:", valid_drawers)
            user_input = input("Enter drawer choice: ").strip()
            if user_input in valid_drawers:
                drawer_choice = user_input
                break
            else:
                print("Invalid choice. Please choose a valid drawer.")

        # --- Check if the chosen objects are present ---
        for obj_name in object_list:
            if obj_name not in positions:
                print(f"[Exploration] Object '{obj_name}' not found in environment. Skipping.")
                continue

            # --- Move to ready-pose if not already there ---
            try:
                robot_pos = None
                for loc in location_list:
                    if f"robot-at_{loc}" in positions and positions[f"robot-at_{loc}"]:
                        robot_pos = loc
                        break
                if robot_pos != "ready-pose":
                    obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location="ready-pose")
            except Exception as e:
                print(f"[Error] Could not move to ready-pose: {e}")

            # --- Pick the object if on the floor ---
            try:
                # Check if object is on the floor
                if f"{obj_name}_on-floor" in positions and positions[f"{obj_name}_on-floor"]:
                    obs, reward, done = execute_pick(env, task, object_name=obj_name, location="floor")
                else:
                    print(f"[Exploration] Object '{obj_name}' is not on the floor or already picked.")
            except Exception as e:
                print(f"[Error] Could not pick '{obj_name}': {e}")

            # --- Move to drawer-area if not already there ---
            try:
                robot_pos = None
                for loc in location_list:
                    if f"robot-at_{loc}" in positions and positions[f"robot-at_{loc}"]:
                        robot_pos = loc
                        break
                if robot_pos != "drawer-area":
                    obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location="drawer-area")
            except Exception as e:
                print(f"[Error] Could not move to drawer-area: {e}")

            # --- Open the drawer if needed ---
            try:
                # Find the handle for the chosen drawer
                handle_map = {
                    "drawer_top": "handle_top",
                    "drawer_middle": "handle_middle",
                    "drawer_bottom": "handle_bottom"
                }
                handle_name = handle_map[drawer_choice]
                # Pick the handle if not already holding
                if f"{handle_name}_on-floor" in positions and positions[f"{handle_name}_on-floor"]:
                    obs, reward, done = execute_pick(env, task, object_name=handle_name, location="floor")
                # Pull the drawer open
                obs, reward, done = execute_pull(env, task, drawer_name=drawer_choice, handle_name=handle_name, location="drawer-area")
            except Exception as e:
                print(f"[Error] Could not open '{drawer_choice}': {e}")

            # --- Place the object in the drawer ---
            try:
                obs, reward, done = execute_place(env, task, object_name=obj_name, drawer_name=drawer_choice, location="drawer-area")
            except Exception as e:
                print(f"[Error] Could not place '{obj_name}' in '{drawer_choice}': {e}")

            # --- Push the drawer closed ---
            try:
                obs, reward, done = execute_push(env, task, drawer_name=drawer_choice, location="drawer-area")
            except Exception as e:
                print(f"[Error] Could not push '{drawer_choice}' closed: {e}")

            # --- Return to ready-pose ---
            try:
                obs, reward, done = execute_go(env, task, from_location="drawer-area", to_location="ready-pose")
            except Exception as e:
                print(f"[Error] Could not return to ready-pose: {e}")

        print("[Task] All objects processed. Task complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
