# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names based on the problem ===
        object_names = ['dice1', 'dice2']
        handle_names = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_names = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # === Exploration Phase: Identify missing predicate ===
        # The feedback suggests we need to check if the object is in the object list before placing.
        # We'll simulate an exploration phase to check for the presence of the object before placing.

        # For this example, let's try to place dice1 in drawer_top, but first check if dice1 is in object_names.
        dice_to_place = 'dice1'
        target_drawer = 'drawer_top'
        target_handle = 'handle_top'
        target_location = 'drawer-area'
        ready_pose = 'ready-pose'
        floor_location = 'floor'

        # Exploration: Check if dice1 is in the object list before proceeding
        if dice_to_place not in object_names:
            print(f"[Exploration] {dice_to_place} not in object list. Aborting placement.")
            return

        # === Task Plan Execution ===
        # 1. Move to ready-pose if not already there
        try:
            robot_pos = positions.get('robot', None)
            if robot_pos is not None and robot_pos != ready_pose:
                obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location=ready_pose)
                print(f"[Task] Robot moved to ready-pose.")
        except Exception as e:
            print(f"[Error] Failed to move to ready-pose: {e}")

        # 2. Pick up dice1 from the floor
        try:
            obs, reward, done = execute_pick(env, task, object_name=dice_to_place, location=floor_location)
            print(f"[Task] Picked up {dice_to_place} from {floor_location}.")
        except Exception as e:
            print(f"[Error] Failed to pick up {dice_to_place}: {e}")
            return

        # 3. Move to drawer-area (if not already there)
        try:
            obs, reward, done = execute_go(env, task, from_location=ready_pose, to_location=target_location)
            print(f"[Task] Robot moved to {target_location}.")
        except Exception as e:
            print(f"[Error] Failed to move to {target_location}: {e}")

        # 4. Place dice1 in drawer_top (drawer must be open and not full)
        # First, open the drawer if needed
        try:
            # Pick up the handle for the drawer
            obs, reward, done = execute_pick(env, task, object_name=target_handle, location=target_location)
            print(f"[Task] Picked up handle {target_handle} at {target_location}.")
        except Exception as e:
            print(f"[Error] Failed to pick up handle {target_handle}: {e}")
            return

        try:
            # Pull the drawer open
            obs, reward, done = execute_pull(env, task, drawer_name=target_drawer, handle_name=target_handle, location=target_location)
            print(f"[Task] Pulled open {target_drawer} using {target_handle}.")
        except Exception as e:
            print(f"[Error] Failed to pull open {target_drawer}: {e}")
            return

        # Now, place the dice in the drawer
        try:
            obs, reward, done = execute_place(env, task, object_name=dice_to_place, drawer_name=target_drawer, location=target_location)
            print(f"[Task] Placed {dice_to_place} in {target_drawer}.")
        except Exception as e:
            print(f"[Error] Failed to place {dice_to_place} in {target_drawer}: {e}")
            return

        # 5. Push the drawer closed
        try:
            obs, reward, done = execute_push(env, task, drawer_name=target_drawer, location=target_location)
            print(f"[Task] Pushed {target_drawer} closed.")
        except Exception as e:
            print(f"[Error] Failed to push {target_drawer} closed: {e}")

        # 6. Return to ready-pose
        try:
            obs, reward, done = execute_go_ready(env, task, from_location=target_location)
            print(f"[Task] Robot returned to ready-pose.")
        except Exception as e:
            print(f"[Error] Failed to return to ready-pose: {e}")

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
