# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names as per the PDDL problem ===
        dice_names = ['dice1', 'dice2']
        handle_names = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_names = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # For this task, we will put dice1 into drawer_bottom using handle_bottom

        dice = 'dice1'
        handle = 'handle_bottom'
        drawer = 'drawer_bottom'
        floor = 'floor'
        drawer_area = 'drawer-area'
        ready_pose = 'ready-pose'

        # === Exploration Phase: Check for missing predicates ===
        # Feedback suggests we need to check if the dice is in the correct location before picking.
        # We'll check if dice1 is on the floor and in the expected area.

        # Get dice1 position and check if it's on the floor (simulate predicate check)
        dice_pos = positions.get(dice, None)
        if dice_pos is None:
            print(f"[Error] {dice} position not found in object positions.")
            return

        # For this simulation, we assume the dice is on the floor in the correct area if present in positions
        # If not, we would need to explore or move to the correct area

        # === Plan Execution ===

        # 1. Ensure robot is at ready-pose
        robot_loc = positions.get('robot', None)
        if robot_loc is None:
            print("[Error] Robot position not found.")
            return

        # If not at ready-pose, move there
        try:
            obs, reward, done, info = execute_go(
                env, task, from_location=robot_loc, to_location=ready_pose
            )
            print(f"[Task] Robot moved to ready-pose.")
        except Exception as e:
            print(f"[Warning] Could not move to ready-pose: {e}")

        # 2. Move to floor (where dice1 is located)
        try:
            obs, reward, done, info = execute_go(
                env, task, from_location=ready_pose, to_location=floor
            )
            print(f"[Task] Robot moved to floor.")
        except Exception as e:
            print(f"[Warning] Could not move to floor: {e}")

        # 3. Pick up dice1 from the floor
        try:
            obs, reward, done, info = execute_pick(
                env, task, object_name=dice, location=floor
            )
            print(f"[Task] Picked up {dice} from floor.")
        except Exception as e:
            print(f"[Error] Could not pick up {dice}: {e}")
            return

        # 4. Return to ready-pose (if required by need-ready flag)
        try:
            obs, reward, done, info = execute_go_ready(
                env, task, from_location=floor
            )
            print(f"[Task] Returned to ready-pose after pick.")
        except Exception as e:
            print(f"[Warning] Could not return to ready-pose: {e}")

        # 5. Move to floor to pick up handle_bottom
        try:
            obs, reward, done, info = execute_go(
                env, task, from_location=ready_pose, to_location=floor
            )
            print(f"[Task] Robot moved to floor to pick handle.")
        except Exception as e:
            print(f"[Warning] Could not move to floor for handle: {e}")

        # 6. Pick up handle_bottom
        try:
            obs, reward, done, info = execute_pick(
                env, task, object_name=handle, location=floor
            )
            print(f"[Task] Picked up {handle} from floor.")
        except Exception as e:
            print(f"[Error] Could not pick up {handle}: {e}")
            return

        # 7. Pull (open) the drawer using handle_bottom
        try:
            obs, reward, done, info = execute_pull(
                env, task, drawer_name=drawer, handle_name=handle, location=floor
            )
            print(f"[Task] Pulled open {drawer} using {handle}.")
        except Exception as e:
            print(f"[Error] Could not pull open {drawer}: {e}")
            return

        # 8. Move to ready-pose (if required)
        try:
            obs, reward, done, info = execute_go_ready(
                env, task, from_location=floor
            )
            print(f"[Task] Returned to ready-pose after pull.")
        except Exception as e:
            print(f"[Warning] Could not return to ready-pose after pull: {e}")

        # 9. Move to drawer-area to place dice1 in drawer_bottom
        try:
            obs, reward, done, info = execute_go(
                env, task, from_location=ready_pose, to_location=drawer_area
            )
            print(f"[Task] Robot moved to drawer-area to place dice.")
        except Exception as e:
            print(f"[Warning] Could not move to drawer-area: {e}")

        # 10. Place dice1 into drawer_bottom
        try:
            obs, reward, done, info = execute_place(
                env, task, object_name=dice, drawer_name=drawer, location=drawer_area
            )
            print(f"[Task] Placed {dice} into {drawer}.")
        except Exception as e:
            print(f"[Error] Could not place {dice} into {drawer}: {e}")
            return

        # 11. Return to ready-pose (if required)
        try:
            obs, reward, done, info = execute_go_ready(
                env, task, from_location=drawer_area
            )
            print(f"[Task] Returned to ready-pose after place.")
        except Exception as e:
            print(f"[Warning] Could not return to ready-pose after place: {e}")

        # 12. Move to floor to push (close) the drawer
        try:
            obs, reward, done, info = execute_go(
                env, task, from_location=ready_pose, to_location=floor
            )
            print(f"[Task] Robot moved to floor to push drawer.")
        except Exception as e:
            print(f"[Warning] Could not move to floor for push: {e}")

        # 13. Push (close) the drawer
        try:
            obs, reward, done, info = execute_push(
                env, task, drawer_name=drawer, location=floor
            )
            print(f"[Task] Pushed (closed) {drawer}.")
        except Exception as e:
            print(f"[Error] Could not push (close) {drawer}: {e}")
            return

        # 14. Return to ready-pose (final)
        try:
            obs, reward, done, info = execute_go_ready(
                env, task, from_location=floor
            )
            print(f"[Task] Returned to ready-pose. Task complete.")
        except Exception as e:
            print(f"[Warning] Could not return to ready-pose at end: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()