# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Get object list from environment ===
        # Assume the environment or task provides a list of object names
        # If not, try to infer from positions or observation
        object_names = []
        if hasattr(task, 'get_object_names'):
            object_names = task.get_object_names()
        elif isinstance(positions, dict):
            object_names = list(positions.keys())
        else:
            # Fallback: try to get from obs or descriptions
            if isinstance(obs, dict) and 'object_names' in obs:
                object_names = obs['object_names']
            elif isinstance(descriptions, dict) and 'object_names' in descriptions:
                object_names = descriptions['object_names']

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to check if an object (e.g., "dice") is in the object list before acting on it.
        # We'll perform an exploration step to ensure all objects are identified and known.
        # This is a generic exploration phase to ensure all objects are known before manipulation.

        # For each object, check if it is present before manipulation
        # For demonstration, let's say we want to manipulate "dice"
        target_object = "dice"
        if target_object not in object_names:
            print(f"[Exploration] Object '{target_object}' not found in object list. Skipping manipulation.")
        else:
            print(f"[Exploration] Object '{target_object}' found. Proceeding with manipulation.")

            # Example: Move to the object's location and identify it
            try:
                dice_pos = positions[target_object]
            except KeyError:
                print(f"[Exploration] Position for '{target_object}' not found. Skipping.")
                dice_pos = None

            if dice_pos is not None:
                # Suppose we have a location name for the dice
                # If not, use a default or infer from positions
                dice_location = None
                if hasattr(task, 'get_location_of_object'):
                    dice_location = task.get_location_of_object(target_object)
                elif isinstance(dice_pos, dict) and 'location' in dice_pos:
                    dice_location = dice_pos['location']
                else:
                    # Fallback: use a default location or skip
                    dice_location = 'table_top'  # Example default

                # Move to the dice location using execute_go
                try:
                    print(f"[Task] Moving to location of '{target_object}': {dice_location}")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location='ready-pose',  # Assume starting from ready-pose
                        to_location=dice_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Error] Failed to move to '{dice_location}': {e}")

                # Pick the dice using execute_pick
                try:
                    print(f"[Task] Picking up '{target_object}' at {dice_location}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=target_object,
                        location=dice_location,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Error] Failed to pick '{target_object}': {e}")

                # Place the dice in the drawer (if drawer exists)
                drawer_name = "drawer_top"
                drawer_location = "drawer_top_place_left"
                if drawer_name in object_names:
                    try:
                        print(f"[Task] Placing '{target_object}' in '{drawer_name}' at '{drawer_location}'")
                        obs, reward, done = execute_place(
                            env,
                            task,
                            object_name=target_object,
                            drawer_name=drawer_name,
                            location=drawer_location,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                    except Exception as e:
                        print(f"[Error] Failed to place '{target_object}' in '{drawer_name}': {e}")
                else:
                    print(f"[Exploration] Drawer '{drawer_name}' not found. Skipping place action.")

        # === General Plan Execution (for all objects) ===
        # Example: For each object, check if it exists before manipulation
        for obj in object_names:
            # Skip if not a manipulable object (e.g., skip handles, places, etc.)
            if obj.startswith("drawer_") or obj == "trash_bin":
                continue

            # Get object position and location
            obj_pos = positions.get(obj, None)
            if obj_pos is None:
                print(f"[Exploration] Position for '{obj}' not found. Skipping.")
                continue

            # Try to get location
            obj_location = None
            if hasattr(task, 'get_location_of_object'):
                obj_location = task.get_location_of_object(obj)
            elif isinstance(obj_pos, dict) and 'location' in obj_pos:
                obj_location = obj_pos['location']
            else:
                obj_location = 'table_top'

            # Move to object
            try:
                print(f"[Task] Moving to '{obj}' at '{obj_location}'")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location='ready-pose',
                    to_location=obj_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] Failed to move to '{obj_location}': {e}")

            # Pick object
            try:
                print(f"[Task] Picking up '{obj}' at '{obj_location}'")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj,
                    location=obj_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] Failed to pick '{obj}': {e}")

            # Place object in drawer if drawer exists
            drawer_name = "drawer_top"
            drawer_location = "drawer_top_place_left"
            if drawer_name in object_names:
                try:
                    print(f"[Task] Placing '{obj}' in '{drawer_name}' at '{drawer_location}'")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_name=obj,
                        drawer_name=drawer_name,
                        location=drawer_location,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Error] Failed to place '{obj}' in '{drawer_name}': {e}")
            else:
                print(f"[Exploration] Drawer '{drawer_name}' not found. Skipping place action.")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
