# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object List for Validation (from feedback) ===
        object_list = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "dice1",
            "dice2",
            "trash",
            "trash_bin"
        ]

        def check_object_validity(object_name):
            return object_name in object_list

        # === Exploration Phase: Discover Missing Predicate ===
        # The goal is to explore the environment and try available skills to discover which predicate is missing.
        # We will attempt to use each available skill and observe the outcome, handling errors gracefully.

        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # For demonstration, we will attempt to use each skill on available objects/positions.
        # In a real scenario, this would be guided by the oracle plan or a specific exploration strategy.

        # Example: Try to pick each object and handle errors if the object is not valid
        for obj_name in object_list:
            if not check_object_validity(obj_name):
                print(f"[Exploration] Object '{obj_name}' is not valid, skipping.")
                continue
            try:
                # Try to pick the object if possible
                # We need to determine the location for the object
                obj_pos = positions.get(obj_name, None)
                if obj_pos is None:
                    print(f"[Exploration] Position for '{obj_name}' not found, skipping.")
                    continue
                print(f"[Exploration] Attempting to pick '{obj_name}' at position {obj_pos}...")
                # Call the pick skill (if available)
                if 'execute_pick' in available_skills:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        target=obj_name,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Successfully picked '{obj_name}'.")
                        break
            except Exception as e:
                print(f"[Exploration] Exception during picking '{obj_name}': {e}")

        # Try to move the robot to each location (if possible)
        for loc_name, loc_pos in positions.items():
            try:
                print(f"[Exploration] Attempting to move to '{loc_name}' at position {loc_pos}...")
                if 'execute_go' in available_skills:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_pos=None,  # If needed, provide current position
                        to_pos=loc_pos,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Successfully moved to '{loc_name}'.")
            except Exception as e:
                print(f"[Exploration] Exception during move to '{loc_name}': {e}")

        # Try to use other skills as part of exploration
        for obj_name in object_list:
            if not check_object_validity(obj_name):
                continue
            obj_pos = positions.get(obj_name, None)
            if obj_pos is None:
                continue
            # Try place, push, pull, sweep, rotate, gripper
            for skill in ['execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_gripper']:
                if skill in available_skills:
                    try:
                        print(f"[Exploration] Attempting skill '{skill}' on '{obj_name}'...")
                        skill_fn = globals()[skill]
                        # The arguments depend on the skill; we use generic ones for demonstration
                        if skill == 'execute_place':
                            obs, reward, done = skill_fn(
                                env,
                                task,
                                target=obj_name,
                                target_pos=obj_pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                        elif skill == 'execute_push':
                            obs, reward, done = skill_fn(
                                env,
                                task,
                                target=obj_name,
                                target_pos=obj_pos,
                                max_steps=100,
                                threshold=0.01,
                                timeout=10.0
                            )
                        elif skill == 'execute_pull':
                            obs, reward, done = skill_fn(
                                env,
                                task,
                                target=obj_name,
                                target_pos=obj_pos,
                                max_steps=100,
                                threshold=0.01,
                                timeout=10.0
                            )
                        elif skill == 'execute_sweep':
                            obs, reward, done = skill_fn(
                                env,
                                task,
                                target=obj_name,
                                target_pos=obj_pos,
                                max_steps=100,
                                threshold=0.01,
                                timeout=10.0
                            )
                        elif skill == 'execute_rotate':
                            obs, reward, done = skill_fn(
                                env,
                                task,
                                target=obj_name,
                                target_pos=obj_pos,
                                max_steps=100,
                                threshold=0.01,
                                timeout=10.0
                            )
                        elif skill == 'execute_gripper':
                            obs, reward, done = skill_fn(
                                env,
                                task,
                                max_steps=100,
                                timeout=10.0
                            )
                        else:
                            continue
                        if done:
                            print(f"[Exploration] Skill '{skill}' on '{obj_name}' succeeded.")
                    except Exception as e:
                        print(f"[Exploration] Exception during '{skill}' on '{obj_name}': {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
