# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Retrieve object and drawer lists from the environment ===
        # These should be available from the environment or task description
        # If not, you may need to parse them from descriptions or obs
        try:
            object_names = task.get_object_names() if hasattr(task, "get_object_names") else []
            drawer_names = task.get_drawer_names() if hasattr(task, "get_drawer_names") else []
            location_names = task.get_location_names() if hasattr(task, "get_location_names") else []
        except Exception as e:
            print("[Warning] Could not retrieve object/drawer/location names from task:", e)
            object_names = []
            drawer_names = []
            location_names = []

        # === Exploration Phase: Identify missing predicates ===
        # The feedback suggests we need to check if an object is in the object list before inserting into the drawer.
        # We simulate an exploration phase to check for object validity before insertion.

        # For demonstration, let's assume the user (or plan) wants to insert a specific object into a drawer.
        # We'll check if the object is valid before proceeding.

        # Example: Try to insert each object in a candidate list (some valid, some invalid)
        candidate_objects = object_names + ["invalid_object_1", "invalid_object_2"]
        target_drawer = drawer_names[0] if drawer_names else None
        target_location = location_names[0] if location_names else None

        for object_to_insert in candidate_objects:
            print(f"[Exploration] Attempting to insert object: {object_to_insert}")
            if object_to_insert in object_names:
                try:
                    # 1. Pick the object (assume it's on the floor at target_location)
                    print(f"[Skill] Picking up {object_to_insert} at {target_location}")
                    obs, reward, done = execute_pick(env, task, object_to_insert, target_location)
                    if done:
                        print(f"[Task] Task ended after picking {object_to_insert}!")
                        break

                    # 2. Place the object into the drawer
                    print(f"[Skill] Placing {object_to_insert} into {target_drawer} at {target_location}")
                    obs, reward, done = execute_place(env, task, object_to_insert, target_drawer, target_location)
                    if done:
                        print(f"[Task] Task ended after placing {object_to_insert}!")
                        break

                    print(f"Object {object_to_insert} inserted into drawer {target_drawer}.")
                except Exception as e:
                    print(f"[Error] Failed to insert {object_to_insert}: {e}")
            else:
                print(f"Invalid object {object_to_insert}. Please insert a valid object.")

        # === Additional Exploration: Use execute_pull to check lock-known predicate ===
        # If the drawer has a handle, try to pull and check lock-known
        for drawer in drawer_names:
            handle_name = None
            # Try to find a handle for this drawer
            for obj in object_names:
                # Assume handle objects are named with 'handle' and associated with the drawer
                if "handle" in obj and drawer in obj:
                    handle_name = obj
                    break
            if handle_name:
                try:
                    print(f"[Exploration] Attempting to pull {drawer} using handle {handle_name} at {target_location}")
                    # Pick the handle first
                    obs, reward, done = execute_pick(env, task, handle_name, target_location)
                    # Pull the drawer
                    obs, reward, done = execute_pull(env, task, drawer, handle_name, target_location)
                    print(f"Pulled {drawer} with {handle_name}.")
                except Exception as e:
                    print(f"[Error] Failed to pull {drawer} with {handle_name}: {e}")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
