# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests that the robot may not know the lock state of the drawer.
        # We need to ensure the robot is aware of the lock state before attempting to open the drawer.
        # We'll use available skills to explore and check the lock state.

        # Example object and location names (should match your environment)
        robot_location = 'ready-pose'
        die_name = 'dice1'
        drawer_name = 'drawer_top'
        drawer_handle = 'drawer_top_handle'
        drawer_location = 'drawer_top_place_left'  # Example location near the drawer
        die_location = 'dice1'  # For simplicity, assume die is at its own name position

        # 1. Move robot to the die's location and identify it (if needed)
        try:
            print("[Exploration] Moving to die location to identify object.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=die_location
            )
            robot_location = die_location
        except Exception as e:
            print(f"[Error] Failed to move to die location: {e}")

        # 2. Pick up the die (if on floor and hand is empty)
        try:
            print("[Task] Picking up the die.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=die_name,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to pick up the die: {e}")

        # 3. Move to the drawer handle location
        try:
            print("[Task] Moving to drawer handle location.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_handle
            )
            robot_location = drawer_handle
        except Exception as e:
            print(f"[Error] Failed to move to drawer handle: {e}")

        # 4. Pick up the drawer handle (if on floor and hand is empty)
        try:
            print("[Task] Picking up the drawer handle.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=drawer_handle,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to pick up the drawer handle: {e}")

        # 5. Exploration: Try to pull the drawer to check lock state (lock-known predicate)
        try:
            print("[Exploration] Attempting to pull the drawer to check lock state.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=drawer_handle,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to pull the drawer (may be locked): {e}")

        # 6. Place the handle back (if required by your domain)
        try:
            print("[Task] Placing the handle back (if required).")
            # If your domain requires placing the handle back, use execute_place
            # obs, reward, done = execute_place(
            #     env,
            #     task,
            #     object_name=drawer_handle,
            #     drawer_name=drawer_name,
            #     location=robot_location
            # )
        except Exception as e:
            print(f"[Error] Failed to place the handle: {e}")

        # 7. Move to the drawer location to place the die
        try:
            print("[Task] Moving to drawer location to place the die.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_location
            )
            robot_location = drawer_location
        except Exception as e:
            print(f"[Error] Failed to move to drawer location: {e}")

        # 8. Place the die into the drawer
        try:
            print("[Task] Placing the die into the drawer.")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=die_name,
                drawer_name=drawer_name,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to place the die into the drawer: {e}")

        # 9. Push the drawer closed
        try:
            print("[Task] Pushing the drawer closed.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to push the drawer closed: {e}")

        # 10. Return to ready pose
        try:
            print("[Task] Returning to ready pose.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location='ready-pose'
            )
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to return to ready pose: {e}")

        print("[Task] Task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()