# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Names (from observation/domain knowledge) ===
        # These should match the actual object names in the environment
        object_names = [
            "dice1", "dice2",
            "handle_bottom", "handle_middle", "handle_top"
        ]
        drawer_names = [
            "drawer_bottom", "drawer_middle", "drawer_top"
        ]
        location_names = [
            "floor", "drawer-area", "ready-pose"
        ]

        # === Force Calibration (Exploration Phase) ===
        # Before executing the main plan, perform force calibration if not already done.
        # This is a placeholder for the exploration phase to discover missing predicates (e.g., force calibration).
        # In a real scenario, this could be a dedicated skill or a check.
        force_calibrated = False
        try:
            # Try to perform a calibration action if available
            # (Assume execute_gripper is used for calibration in this context)
            print("[Exploration] Performing force calibration using execute_gripper...")
            obs, reward, done = execute_gripper(env, task)
            force_calibrated = True
            print("[Exploration] Force calibration completed.")
        except Exception as e:
            print("[Exploration] Force calibration skill not available or failed:", e)
            # Proceed, but warn that force calibration may be missing

        # === Object Validation (Exploration Phase) ===
        # Ensure all objects exist in the environment before proceeding
        missing_objects = []
        for obj in object_names:
            if obj not in positions:
                missing_objects.append(obj)
        if missing_objects:
            print("[Validation] Warning: The following objects are missing in the environment:", missing_objects)
            # Optionally, skip actions involving missing objects

        # === Main Task Plan ===
        # Example: Put dice1 and dice2 into drawer_bottom, using handles as needed, and return to ready-pose

        # 1. Move to ready-pose if not already there
        try:
            robot_pos = None
            for loc in location_names:
                if positions.get("robot", None) == loc:
                    robot_pos = loc
                    break
            if robot_pos != "ready-pose":
                print("[Task] Moving robot to ready-pose...")
                obs, reward, done = execute_go(env, task, robot_pos, "ready-pose")
        except Exception as e:
            print("[Task] Could not determine robot position or move to ready-pose:", e)

        # 2. For each dice, pick and place into drawer_bottom
        for dice in ["dice1", "dice2"]:
            if dice not in positions:
                print(f"[Task] Skipping {dice} as it is missing from the environment.")
                continue

            # a) Move to dice location if not already there
            dice_pos = positions[dice]
            try:
                robot_pos = positions.get("robot", "ready-pose")
                if robot_pos != dice_pos:
                    print(f"[Task] Moving to {dice} at {dice_pos}...")
                    obs, reward, done = execute_go(env, task, robot_pos, dice_pos)
            except Exception as e:
                print(f"[Task] Could not move to {dice} location:", e)
                continue

            # b) Pick the dice
            try:
                print(f"[Task] Picking up {dice}...")
                obs, reward, done = execute_pick(env, task, dice, dice_pos)
            except Exception as e:
                print(f"[Task] Failed to pick {dice}:", e)
                continue

            # c) Move to drawer-area (assume drawer-area is the location for drawer_bottom)
            try:
                robot_pos = positions.get("robot", dice_pos)
                if robot_pos != "drawer-area":
                    print("[Task] Moving to drawer-area for placing dice...")
                    obs, reward, done = execute_go(env, task, robot_pos, "drawer-area")
            except Exception as e:
                print("[Task] Could not move to drawer-area:", e)
                continue

            # d) Open the drawer if needed (pull handle)
            try:
                # Pick handle if not already holding
                print("[Task] Picking handle_bottom to open drawer_bottom...")
                obs, reward, done = execute_pick(env, task, "handle_bottom", "drawer-area")
                print("[Task] Pulling drawer_bottom using handle_bottom...")
                obs, reward, done = execute_pull(env, task, "drawer_bottom", "handle_bottom", "drawer-area")
            except Exception as e:
                print("[Task] Could not open drawer_bottom:", e)
                # Continue anyway

            # e) Place the dice into the drawer
            try:
                print(f"[Task] Placing {dice} into drawer_bottom...")
                obs, reward, done = execute_place(env, task, dice, "drawer_bottom", "drawer-area")
            except Exception as e:
                print(f"[Task] Failed to place {dice} into drawer_bottom:", e)
                continue

            # f) Return to ready-pose (if required by need-ready)
            try:
                print("[Task] Returning to ready-pose after placing object...")
                obs, reward, done = execute_go_ready(env, task, "drawer-area")
            except Exception as e:
                print("[Task] Could not return to ready-pose:", e)

        # 3. Push the drawer closed
        try:
            print("[Task] Pushing drawer_bottom closed...")
            obs, reward, done = execute_push(env, task, "drawer_bottom", "drawer-area")
        except Exception as e:
            print("[Task] Could not push drawer_bottom closed:", e)

        # 4. Final ready-pose
        try:
            print("[Task] Moving to final ready-pose...")
            obs, reward, done = execute_go(env, task, "drawer-area", "ready-pose")
        except Exception as e:
            print("[Task] Could not move to final ready-pose:", e)

        print("[Task] Task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
