# run_skeleton_task.py (Completed with Exploration and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names ===
        # These should match the environment's naming conventions
        die_to_insert = "dice1"
        drawer_name = "drawer_middle"
        drawer_handle = "drawer_middle_handle"
        drawer_location = "drawer_middle_place"
        ready_pose = "ready-pose"

        # For exploration, we need to check for missing predicates (e.g., is the die in the drawer?)
        # We'll use the available skills to perform the task and then check the result.

        # --- 1. Move to the die's location ---
        try:
            die_pos = positions[die_to_insert]
        except KeyError:
            print(f"[Error] Die '{die_to_insert}' not found in object positions.")
            return

        # Find the robot's current location
        robot_location = None
        for obj, pos in positions.items():
            if obj.startswith("robot") or obj == "robot":
                robot_location = pos
                break

        # If robot location is not found, assume starting at ready-pose
        if robot_location is None:
            robot_location = positions.get(ready_pose, None)

        # --- 2. Move to die location if not already there ---
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=die_pos
            )
            robot_location = die_pos
        except Exception as e:
            print(f"[Error] Failed to move to die location: {e}")
            return

        # --- 3. Pick up the die ---
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=die_to_insert,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to pick up die: {e}")
            return

        # --- 4. Move to drawer location ---
        try:
            drawer_pos = positions[drawer_location]
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_pos
            )
            robot_location = drawer_pos
        except Exception as e:
            print(f"[Error] Failed to move to drawer location: {e}")
            return

        # --- 5. Open the drawer (pull handle) ---
        try:
            # Pick up the handle if needed
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=drawer_handle,
                location=robot_location
            )
            # Pull the drawer open
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=drawer_handle,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to open drawer: {e}")
            return

        # --- 6. Place the die in the drawer ---
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=die_to_insert,
                drawer_name=drawer_name,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to place die in drawer: {e}")
            return

        # --- 7. Push the drawer closed ---
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=robot_location
            )
        except Exception as e:
            print(f"[Error] Failed to push drawer closed: {e}")
            return

        # --- 8. Return to ready pose ---
        try:
            ready_pos = positions.get(ready_pose, None)
            if ready_pos is not None and robot_location != ready_pos:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=ready_pos
                )
        except Exception as e:
            print(f"[Error] Failed to return to ready pose: {e}")

        # === Feedback/Exploration: Check if die is in the drawer ===
        # This is the key feedback-driven check: is the die in the drawer?
        # We'll use the environment's observation or positions to check this.
        try:
            # Refresh positions after actions
            positions = get_object_positions()
            # Check if die is now in the drawer
            die_in_drawer = False
            # Some environments may encode this as a property or location
            die_location = positions.get(die_to_insert, None)
            if die_location is not None:
                # If the die's location matches the drawer's location, assume it's inside
                if die_location == drawer_location or die_location == drawer_name:
                    die_in_drawer = True
            # Alternatively, check observation predicates if available
            # (e.g., obs['in-drawer'][die_to_insert] == drawer_name)
            if die_in_drawer:
                print("Die inserted successfully.")
            else:
                print("Die not found in drawer. Please try again.")
        except Exception as e:
            print(f"[Error] Could not verify die in drawer: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
