# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === OBJECT AND LOCATION NAMES (from feedback and PDDL) ===
        dice_name = 'dice1'
        handle_name = 'handle_bottom'
        drawer_name = 'drawer_bottom'
        ready_pose = 'ready-pose'
        floor = 'floor'
        drawer_area = 'drawer-area'

        # === PLAN: Store dice1 inside drawer_bottom ===
        # 1. Move to dice1 (on floor)
        # 2. Pick up dice1
        # 3. Go to ready-pose (if needed)
        # 4. Move to drawer area
        # 5. Pick up handle_bottom
        # 6. Pull to open drawer_bottom
        # 7. Go to ready-pose (if needed)
        # 8. Place dice1 in drawer_bottom
        # 9. Go to ready-pose (if needed)
        # 10. Push to close drawer_bottom

        # === STEP 1: Move to dice1 (on floor) ===
        try:
            obs, reward, done, info = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=floor
            )
            print("[Task] Moved to floor for dice1.")
        except Exception as e:
            print(f"[Error] Failed to move to floor: {e}")
            return

        # === STEP 2: Pick up dice1 ===
        try:
            obs, reward, done, info = execute_pick(
                env,
                task,
                object_name=dice_name,
                location=floor
            )
            print("[Task] Picked up dice1.")
        except Exception as e:
            print(f"[Error] Failed to pick up dice1: {e}")
            return

        # === STEP 3: Go to ready-pose (required after pick) ===
        try:
            obs, reward, done, info = execute_go_ready(
                env,
                task,
                from_location=floor
            )
            print("[Task] Returned to ready-pose after picking dice1.")
        except Exception as e:
            print(f"[Error] Failed to return to ready-pose: {e}")
            return

        # === STEP 4: Move to drawer area ===
        try:
            obs, reward, done, info = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=drawer_area
            )
            print("[Task] Moved to drawer area.")
        except Exception as e:
            print(f"[Error] Failed to move to drawer area: {e}")
            return

        # === STEP 5: Pick up handle_bottom ===
        try:
            obs, reward, done, info = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=drawer_area
            )
            print("[Task] Picked up handle_bottom.")
        except Exception as e:
            print(f"[Error] Failed to pick up handle_bottom: {e}")
            return

        # === STEP 6: Pull to open drawer_bottom ===
        try:
            obs, reward, done, info = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_area
            )
            print("[Task] Pulled handle to open drawer_bottom.")
        except Exception as e:
            print(f"[Error] Failed to pull and open drawer: {e}")
            return

        # === STEP 7: Go to ready-pose (required after pull) ===
        try:
            obs, reward, done, info = execute_go_ready(
                env,
                task,
                from_location=drawer_area
            )
            print("[Task] Returned to ready-pose after opening drawer.")
        except Exception as e:
            print(f"[Error] Failed to return to ready-pose: {e}")
            return

        # === STEP 8: Move to drawer area (again, to place dice) ===
        try:
            obs, reward, done, info = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=drawer_area
            )
            print("[Task] Moved to drawer area to place dice.")
        except Exception as e:
            print(f"[Error] Failed to move to drawer area for placing dice: {e}")
            return

        # === STEP 9: Place dice1 in drawer_bottom ===
        try:
            obs, reward, done, info = execute_place(
                env,
                task,
                object_name=dice_name,
                drawer_name=drawer_name,
                location=drawer_area
            )
            print("[Task] Placed dice1 in drawer_bottom.")
        except Exception as e:
            print(f"[Error] Failed to place dice1 in drawer: {e}")
            return

        # === STEP 10: Go to ready-pose (required after place) ===
        try:
            obs, reward, done, info = execute_go_ready(
                env,
                task,
                from_location=drawer_area
            )
            print("[Task] Returned to ready-pose after placing dice.")
        except Exception as e:
            print(f"[Error] Failed to return to ready-pose after placing dice: {e}")
            return

        # === STEP 11: Move to drawer area to push drawer closed ===
        try:
            obs, reward, done, info = execute_go(
                env,
                task,
                from_location=ready_pose,
                to_location=drawer_area
            )
            print("[Task] Moved to drawer area to push drawer closed.")
        except Exception as e:
            print(f"[Error] Failed to move to drawer area for push: {e}")
            return

        # === STEP 12: Push to close drawer_bottom ===
        try:
            obs, reward, done, info = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_area
            )
            print("[Task] Pushed drawer_bottom closed.")
        except Exception as e:
            print(f"[Error] Failed to push and close drawer: {e}")
            return

        print("[Task] Successfully stored dice1 inside drawer_bottom.")

        # === EXPLORATION PHASE: Identify missing predicate if task fails ===
        # If at any point above, the plan fails, we could try to explore the environment
        # to check for missing predicates (e.g., lock-known, identified, etc.)
        # For demonstration, let's check if the drawer is locked and, if so, try to identify lock state

        # (This is a placeholder for exploration logic; in a real system, you would check the state)
        # For example, if execute_pull fails due to lock, you might need to call an exploration skill
        # such as execute_pull (from exploration domain) to set lock-known predicate.

        # Example exploration (pseudo-code, not executed unless needed):
        # try:
        #     obs, reward, done, info = execute_pull(
        #         env,
        #         task,
        #         object_name=drawer_name,
        #         location=drawer_area
        #     )
        #     print("[Exploration] Checked lock state of drawer.")
        # except Exception as e:
        #     print(f"[Exploration] Could not check lock state: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
