# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and drawer names as per the problem ===
        objects = ['dice1', 'dice2']
        handles = {
            'drawer_bottom': 'handle_bottom',
            'drawer_middle': 'handle_middle',
            'drawer_top': 'handle_top'
        }
        drawers = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        locations = ['floor', 'drawer-area', 'ready-pose']

        # === Helper: Check object and drawer availability ===
        def is_object_available(obj):
            return obj in positions

        def is_drawer_unlocked(drawer):
            # For this simulation, assume we can check the state via positions or task state
            # If not available, assume unlocked for bottom/top, locked for middle as per initial state
            if drawer == 'drawer_middle':
                return False
            return True

        # === Exploration Phase: Identify missing predicate ===
        # The feedback suggests a missing predicate related to object availability and compatibility with the drawer.
        # We'll perform an exploration step to check which predicate is missing by attempting to pick and place.

        # Try to pick and place each object into each drawer, and catch errors to infer missing predicates.
        for obj in objects:
            if not is_object_available(obj):
                print(f"[Exploration] Object {obj} not available in environment. Skipping.")
                continue
            for drawer in drawers:
                print(f"[Exploration] Attempting to place {obj} into {drawer}...")
                # Check if drawer is unlocked before proceeding
                if not is_drawer_unlocked(drawer):
                    print(f"[Exploration] Drawer {drawer} is locked. Cannot place {obj}.")
                    continue
                # Move to object location (assume on floor)
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location='ready-pose',
                        to_location='floor'
                    )
                except Exception as e:
                    print(f"[Exploration] Failed to move to floor for {obj}: {e}")
                    continue
                # Pick the object
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=obj,
                        location='floor'
                    )
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj}: {e}")
                    continue
                # Move to drawer area
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location='floor',
                        to_location='drawer-area'
                    )
                except Exception as e:
                    print(f"[Exploration] Failed to move to drawer-area for {obj}: {e}")
                    continue
                # Open the drawer (pull handle)
                handle = handles[drawer]
                try:
                    # Pick the handle
                    obs, reward, done = execute_pick_handle(
                        env,
                        task,
                        handle_name=handle,
                        location='drawer-area'
                    )
                    # Pull the drawer open
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=drawer,
                        handle_name=handle,
                        location='drawer-area'
                    )
                except Exception as e:
                    print(f"[Exploration] Failed to open {drawer} with {handle}: {e}")
                    continue
                # Place the object in the drawer
                try:
                    obs, reward, done = execute_place(
                        env,
                        task,
                        object_name=obj,
                        drawer_name=drawer,
                        location='drawer-area'
                    )
                    print(f"[Exploration] Successfully placed {obj} in {drawer}.")
                except Exception as e:
                    print(f"[Exploration] Failed to place {obj} in {drawer}: {e}")
                    continue
                # Push the drawer closed
                try:
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name=drawer,
                        location='drawer-area'
                    )
                except Exception as e:
                    print(f"[Exploration] Failed to push {drawer} closed: {e}")
                # Return to ready pose
                try:
                    obs, reward, done = execute_go_ready(
                        env,
                        task,
                        from_location='drawer-area'
                    )
                except Exception as e:
                    print(f"[Exploration] Failed to return to ready-pose: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
