# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (example, adapt as needed) ===
        # These names should match those in your environment and PDDL
        dice_name = "dice1"
        drawer_name = "drawer_middle"
        handle_name = "drawer_middle_handle"
        robot_ready_pose = "ready-pose"
        dice_location = positions.get(dice_name, None)
        handle_location = positions.get(handle_name, None)
        drawer_location = positions.get(drawer_name, None)
        # Fallbacks if not found
        if dice_location is None:
            print("[Error] Dice object not found in positions. Exploration required.")
        if handle_location is None:
            print("[Error] Handle object not found in positions. Exploration required.")
        if drawer_location is None:
            print("[Error] Drawer object not found in positions. Exploration required.")

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to ensure the dice is in the object list and can be manipulated.
        # We also need to check for missing predicates (e.g., lock-known, identified, etc.)
        # Use exploration skills to gather missing information if needed.

        # Example: Try to identify the dice if not already known
        try:
            # If dice is not in object_list, attempt to identify it
            if dice_location is None:
                print("[Exploration] Attempting to identify dice location using execute_go_identify.")
                # Assume robot starts at ready-pose or some default location
                robot_start_loc = "ready-pose"
                # Try all known locations to find dice
                for loc_name, loc_pos in positions.items():
                    try:
                        obs, reward, done = execute_go_identify(
                            env,
                            task,
                            robot=0,  # assuming single robot, id 0
                            from_location=robot_start_loc,
                            to_location=loc_name
                        )
                        # After identification, update positions
                        positions = get_object_positions()
                        if dice_name in positions:
                            dice_location = positions[dice_name]
                            print(f"[Exploration] Dice found at {loc_name}.")
                            break
                    except Exception as e:
                        print(f"[Exploration] Failed to identify at {loc_name}: {e}")
        except Exception as e:
            print(f"[Exploration] Error during identification: {e}")

        # === Safety Checks and Force Calibration (if required) ===
        # For this simulation, we assume the skill functions handle safety and force internally.

        # === Main Task Plan Execution ===
        # 1. Move to dice location
        # 2. Pick up the dice
        # 3. Move to handle location
        # 4. Pick up the handle
        # 5. Pull the drawer open
        # 6. Move to ready pose (if need-ready is set)
        # 7. Place the dice in the drawer
        # 8. Move to ready pose (if need-ready is set)
        # 9. Push the drawer closed

        try:
            # 1. Move to dice location
            print("[Task] Moving to dice location.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_ready_pose,
                to_location=dice_name  # assuming location name matches object
            )

            # 2. Pick up the dice
            print("[Task] Picking up the dice.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=dice_name,
                location=dice_name  # assuming dice is at its own location
            )

            # 3. Move to ready pose (if need-ready is set)
            print("[Task] Moving to ready pose after pick (if required).")
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=dice_name
                )
            except Exception as e:
                print("[Task] execute_go_ready not required or failed:", e)

            # 4. Move to handle location
            print("[Task] Moving to handle location.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_ready_pose,
                to_location=handle_name
            )

            # 5. Pick up the handle
            print("[Task] Picking up the handle.")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=handle_name
            )

            # 6. Pull the drawer open
            print("[Task] Pulling the drawer open.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=handle_name
            )

            # 7. Move to ready pose (if need-ready is set)
            print("[Task] Moving to ready pose after pull (if required).")
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=handle_name
                )
            except Exception as e:
                print("[Task] execute_go_ready not required or failed:", e)

            # 8. Move to drawer location
            print("[Task] Moving to drawer location.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_ready_pose,
                to_location=drawer_name
            )

            # 9. Place the dice in the drawer
            print("[Task] Placing the dice in the drawer.")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=dice_name,
                drawer_name=drawer_name,
                location=drawer_name
            )

            # 10. Move to ready pose (if need-ready is set)
            print("[Task] Moving to ready pose after place (if required).")
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=drawer_name
                )
            except Exception as e:
                print("[Task] execute_go_ready not required or failed:", e)

            # 11. Push the drawer closed
            print("[Task] Pushing the drawer closed.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_name
            )

            print("[Task] Task completed successfully.")

        except Exception as e:
            print(f"[Task] Exception during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
