# run_skeleton_task.py (Filled: Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # DO NOT redefine any primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()  # {name: (x, y, z)}
        
        # In a realistic scenario, you'd likely have access to mappings such as:
        # - object_names: list of object keys
        # - drawer_names: list of drawer keys
        # - location_names: list of possible robot positions
        
        # However, for exploration, our goal is to probe for *missing predicates* by
        # systematically attempting to observe effects distinctly caused by each "exploration action"
        
        # === Exploration Logic for Missing Predicate ===
        # Try each exploration-oriented skill in succession and inspect environment response.

        # List of exploration skills and their likely required arguments
        exploration_skills = [
            ("execute_go_identify", ["from_location", "to_location"]),
            ("execute_go_temperature", ["from_location", "to_location"]),
            ("execute_pick_weight", ["object_name", "at_location"]),
            ("execute_pick_durability", ["object_name", "at_location"]),
            ("execute_pull", ["object_name", "at_location"])  # May need holding as precondition!
        ]
        
        # To keep it safe and conformant, we will attempt the following:
        # - For each skill, try to call it if arguments are available.
        # - Catch exceptions and print out which predicate appears to be missing (from observation/feedback).

        # Try to extract at least one valid location, object for testing.
        object_candidates = [name for name in positions if 'drawer' not in name and 'handle' not in name]  # e.g., object_1
        location_candidates = [name for name in positions if 'location' in name or 'pose' in name or 'room' in name]
        drawer_candidates = [name for name in positions if 'drawer' in name]  # e.g., drawer_1

        # Try to pick a plausible object/location (fallback if names are unknown)
        test_object = object_candidates[0] if object_candidates else next(iter(positions))
        test_location = location_candidates[0] if location_candidates else None

        # For exploration, collect output for each try
        print("\n=== Exploration Phase: Discovering Missing Predicate(s) ===")
        for skill_name, args in exploration_skills:
            print(f"\n[Exploration] Attempting skill: {skill_name}")
            try:
                # Build arguments as per skill signature
                # Arg heuristics:
                if skill_name.startswith("execute_go"):  # move-type skills
                    from_location = test_location if test_location else "ready-pose"
                    # Simulate movement to a second key location
                    to_location = None
                    for k in positions.keys():
                        if k != from_location and (('location' in k) or ('pose' in k) or ('room' in k)):
                            to_location = k
                            break
                    if not to_location:
                        # Fallback: use any two unique locations
                        keys = list(positions.keys())
                        to_location = keys[0] if len(keys) > 0 else from_location
                    # Try calling the move/identify/temperature skills
                    skill_func = globals()[skill_name]
                    obs, reward, done = skill_func(env, task, from_location, to_location)
                    print(f"Result: {obs}, {reward}, {done}")

                elif skill_name in ["execute_pick_weight", "execute_pick_durability"]:
                    obj = test_object
                    # Try to find its location
                    at_location = None
                    for loc_name in positions:
                        if loc_name != obj and ('location' in loc_name or 'pose' in loc_name or 'room' in loc_name):
                            at_location = loc_name
                            break
                    if not at_location:
                        at_location = test_location
                    skill_func = globals()[skill_name]
                    obs, reward, done = skill_func(env, task, obj, at_location)
                    print(f"Result: {obs}, {reward}, {done}")

                elif skill_name == "execute_pull":
                    obj = test_object
                    # Find a plausible location (where object is)
                    at_location = test_location
                    if not at_location:
                        for loc in positions:
                            if loc != obj:
                                at_location = loc
                                break
                    # If the skill expects to be holding the object – simulate basic pick first
                    try:
                        execute_pick(env, task, obj, at_location)
                    except Exception as ee:
                        print(f"Couldn't execute_pick before pull: {ee}")
                    skill_func = globals()[skill_name]
                    obs, reward, done = skill_func(env, task, obj, at_location)
                    print(f"Result: {obs}, {reward}, {done}")

                else:
                    print(f"[Exploration] Skill arguments unclear for: {skill_name}")
                    continue

            except Exception as e:
                # Print out error, which should suggest missing precondition or effect (predicate)
                print(f"[Exploration] Exception for {skill_name}: {e}")

        print("\n=== Exploration Complete. Check logs for missing predicates, preconditions, or effects. ===")

        # After exploration, further task execution can be added here according to the plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
