# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names based on the problem description ===
        dice_names = ['dice1', 'dice2']
        handle_names = ['handle_bottom', 'handle_middle', 'handle_top']
        drawer_names = ['drawer_bottom', 'drawer_middle', 'drawer_top']
        location_names = ['floor', 'drawer-area', 'ready-pose']

        # === Helper state tracking ===
        # We'll keep track of the robot's state to avoid logic errors
        robot_state = {
            'holding': None,
            'at': 'ready-pose',
            'hand_empty': True,
            'robot_free': True,
            'need_ready': False
        }
        # Track drawer states
        drawer_state = {
            'drawer_bottom': {'locked': False, 'open': False, 'full': False, 'empty': True, 'partially_filled': False},
            'drawer_middle': {'locked': False, 'open': False, 'full': False, 'empty': True, 'partially_filled': False},
            'drawer_top': {'locked': False, 'open': False, 'full': False, 'empty': True, 'partially_filled': False}
        }
        # Track which objects are on the floor
        object_on_floor = {name: True for name in dice_names + handle_names}

        # === Exploration Phase: Find the missing predicate ===
        # The feedback suggests we need to explore to determine which predicate is missing.
        # We'll use the available skills to try to identify, weigh, or check lock status of objects.
        # For demonstration, we'll try to "pull" each handle to see if lock-known is set.

        print("[Exploration] Checking for missing predicates (e.g., lock-known) on handles...")
        for handle, drawer in zip(handle_names, drawer_names):
            try:
                # Move to handle location if not already there
                handle_pos = positions.get(handle, None)
                if handle_pos is None:
                    print(f"[Exploration] Warning: Position for {handle} not found.")
                    continue
                if robot_state['at'] != 'drawer-area':
                    obs, reward, done = execute_go(env, task, from_location=robot_state['at'], to_location='drawer-area')
                    robot_state['at'] = 'drawer-area'
                # Pick up the handle if not already holding
                if robot_state['holding'] is None and object_on_floor[handle]:
                    obs, reward, done = execute_pick_handle(env, task, handle, 'drawer-area')
                    robot_state['holding'] = handle
                    robot_state['hand_empty'] = False
                    robot_state['robot_free'] = False
                    object_on_floor[handle] = False
                # Try to pull the handle to check lock-known
                obs, reward, done = execute_pull(env, task, drawer, handle, 'drawer-area')
                print(f"[Exploration] Pulled {handle} of {drawer} to check lock status.")
                # After pull, hand is empty
                robot_state['holding'] = None
                robot_state['hand_empty'] = True
                robot_state['robot_free'] = True
            except Exception as e:
                print(f"[Exploration] Exception during exploration with {handle}: {e}")

        # === Main Task Plan ===
        # For each dice, pick it up and place it into the corresponding drawer
        # Ensure all safety and logic checks as per feedback

        for dice, drawer, handle in zip(dice_names, drawer_names, handle_names):
            try:
                # 1. Move to dice location if not already there
                dice_pos = positions.get(dice, None)
                if dice_pos is None:
                    print(f"[Task] Warning: Position for {dice} not found.")
                    continue
                if robot_state['at'] != 'floor':
                    obs, reward, done = execute_go(env, task, from_location=robot_state['at'], to_location='floor')
                    robot_state['at'] = 'floor'
                # 2. Check if hand is empty before picking
                if not robot_state['hand_empty']:
                    print(f"[Task] Hand not empty before picking {dice}, going to ready pose.")
                    obs, reward, done = execute_go_ready(env, task, from_location=robot_state['at'])
                    robot_state['at'] = 'ready-pose'
                    robot_state['need_ready'] = False
                    robot_state['hand_empty'] = True
                    robot_state['robot_free'] = True
                # 3. Pick the dice
                if object_on_floor[dice]:
                    obs, reward, done = execute_pick_object(env, task, dice, 'floor')
                    robot_state['holding'] = dice
                    robot_state['hand_empty'] = False
                    robot_state['robot_free'] = False
                    robot_state['need_ready'] = True
                    object_on_floor[dice] = False
                # 4. Go to ready pose if need_ready is set
                if robot_state['need_ready']:
                    obs, reward, done = execute_go_ready(env, task, from_location=robot_state['at'])
                    robot_state['at'] = 'ready-pose'
                    robot_state['need_ready'] = False
                    robot_state['hand_empty'] = True
                    robot_state['robot_free'] = True
                # 5. Move to drawer area
                if robot_state['at'] != 'drawer-area':
                    obs, reward, done = execute_go(env, task, from_location=robot_state['at'], to_location='drawer-area')
                    robot_state['at'] = 'drawer-area'
                # 6. Check if drawer is open; if not, open it
                if not drawer_state[drawer]['open']:
                    # Pick the handle if not already holding
                    if robot_state['holding'] is None and object_on_floor[handle]:
                        obs, reward, done = execute_pick_handle(env, task, handle, 'drawer-area')
                        robot_state['holding'] = handle
                        robot_state['hand_empty'] = False
                        robot_state['robot_free'] = False
                        object_on_floor[handle] = False
                    # Pull the handle to open the drawer
                    obs, reward, done = execute_pull(env, task, drawer, handle, 'drawer-area')
                    print(f"[Task] Pulled {handle} to open {drawer}.")
                    robot_state['holding'] = None
                    robot_state['hand_empty'] = True
                    robot_state['robot_free'] = True
                    drawer_state[drawer]['open'] = True
                    drawer_state[drawer]['closed'] = False
                # 7. Check if holding the dice before placing
                if robot_state['holding'] != dice:
                    print(f"[Task] Not holding {dice} before placing, skipping.")
                    continue
                # 8. Place the dice into the drawer
                obs, reward, done = execute_place_object(env, task, dice, drawer, 'drawer-area')
                print(f"[Task] Placed {dice} into {drawer}.")
                robot_state['holding'] = None
                robot_state['hand_empty'] = True
                robot_state['robot_free'] = True
                robot_state['need_ready'] = True
                # Update drawer fill state
                if drawer_state[drawer]['empty']:
                    drawer_state[drawer]['empty'] = False
                    drawer_state[drawer]['partially_filled'] = True
                elif drawer_state[drawer]['partially_filled']:
                    drawer_state[drawer]['partially_filled'] = False
                    drawer_state[drawer]['full'] = True
                # 9. Go to ready pose after placing
                if robot_state['need_ready']:
                    obs, reward, done = execute_go_ready(env, task, from_location=robot_state['at'])
                    robot_state['at'] = 'ready-pose'
                    robot_state['need_ready'] = False
                    robot_state['hand_empty'] = True
                    robot_state['robot_free'] = True
            except Exception as e:
                print(f"[Task] Exception during manipulation of {dice}: {e}")

        print("[Task] All dice placed in drawers. Task complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
