# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object List (from feedback, cleaned) ===
        object_names = [
            "dice1",
            "dice2",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "trash",
            "trash_bin"
        ]
        # Removed "drawer_top_handle" as per feedback

        # === Task: Place one dice into a drawer ===
        # We'll pick the first available dice and the first available drawer

        dice_candidates = [name for name in object_names if name.startswith("dice")]
        drawer_handles = [name for name in object_names if "handle" in name]
        drawer_places = [name for name in object_names if "place" in name and "drawer" in name]

        # Safety: Check that at least one dice and one drawer handle exist
        if not dice_candidates:
            print("[Error] No dice objects found in object list.")
            return
        if not drawer_handles:
            print("[Error] No drawer handle objects found in object list.")
            return

        # Select dice and drawer handle
        dice_to_place = dice_candidates[0]
        drawer_handle = drawer_handles[0]

        # For demonstration, assume the drawer associated with the handle is the same (e.g., "drawer_middle_handle" -> "drawer_middle")
        drawer_name = drawer_handle.replace("_handle", "")
        # Find a place location for this drawer
        drawer_place_candidates = [p for p in drawer_places if drawer_name in p]
        if drawer_place_candidates:
            drawer_place = drawer_place_candidates[0]
        else:
            drawer_place = None

        # === Exploration Phase: Identify Missing Predicate ===
        # Try to pull the drawer; if it fails, check if the drawer is locked (simulate exploration)
        # In a real system, this would be more elaborate, but here we just log the attempt

        print(f"[Exploration] Attempting to pull {drawer_handle} to check lock status...")

        try:
            # Try to pick the handle
            obs, reward, done = execute_pick_handle(
                env, task, obj=dice_to_place, location=positions.get(dice_to_place, None)
            )
        except Exception as e:
            print(f"[Exploration] Could not pick dice for handle test: {e}")

        try:
            obs, reward, done = execute_pick_handle(
                env, task, obj=drawer_handle, location=positions.get(drawer_handle, None)
            )
            print(f"[Exploration] Picked handle {drawer_handle}.")
        except Exception as e:
            print(f"[Exploration] Could not pick handle {drawer_handle}: {e}")

        try:
            obs, reward, done = execute_pull(
                env, task, d=drawer_name, h=drawer_handle, p=positions.get(drawer_handle, None)
            )
            print(f"[Exploration] Pulled {drawer_handle} to open {drawer_name}.")
        except Exception as e:
            print(f"[Exploration] Could not pull {drawer_handle}: {e}")
            print("[Exploration] Drawer may be locked or missing predicate. Exploration needed.")

        # === Main Task Execution ===
        # 1. Go to dice location
        dice_pos = positions.get(dice_to_place, None)
        if dice_pos is None:
            print(f"[Error] Could not find position for {dice_to_place}.")
            return

        try:
            obs, reward, done = execute_go(
                env, task, from_location="ready-pose", to_location=dice_pos
            )
            print(f"[Task] Moved to {dice_to_place} at {dice_pos}.")
        except Exception as e:
            print(f"[Error] Could not move to {dice_to_place}: {e}")
            return

        # 2. Pick the dice
        try:
            obs, reward, done = execute_pick_object(
                env, task, o=dice_to_place, p=dice_pos
            )
            print(f"[Task] Picked up {dice_to_place}.")
        except Exception as e:
            print(f"[Error] Could not pick {dice_to_place}: {e}")
            return

        # 3. Go to drawer handle location
        handle_pos = positions.get(drawer_handle, None)
        if handle_pos is None:
            print(f"[Error] Could not find position for {drawer_handle}.")
            return

        try:
            obs, reward, done = execute_go(
                env, task, from_location=dice_pos, to_location=handle_pos
            )
            print(f"[Task] Moved to {drawer_handle} at {handle_pos}.")
        except Exception as e:
            print(f"[Error] Could not move to {drawer_handle}: {e}")
            return

        # 4. Pick the drawer handle
        try:
            obs, reward, done = execute_pick_handle(
                env, task, h=drawer_handle, p=handle_pos
            )
            print(f"[Task] Picked up handle {drawer_handle}.")
        except Exception as e:
            print(f"[Error] Could not pick handle {drawer_handle}: {e}")
            return

        # 5. Pull the drawer open
        try:
            obs, reward, done = execute_pull(
                env, task, d=drawer_name, h=drawer_handle, p=handle_pos
            )
            print(f"[Task] Pulled open {drawer_name}.")
        except Exception as e:
            print(f"[Error] Could not pull open {drawer_name}: {e}")
            return

        # 6. Go to drawer place location
        if drawer_place is not None:
            place_pos = positions.get(drawer_place, None)
            if place_pos is None:
                print(f"[Error] Could not find position for {drawer_place}.")
                return
            try:
                obs, reward, done = execute_go(
                    env, task, from_location=handle_pos, to_location=place_pos
                )
                print(f"[Task] Moved to drawer place {drawer_place} at {place_pos}.")
            except Exception as e:
                print(f"[Error] Could not move to {drawer_place}: {e}")
                return
        else:
            print("[Warning] No drawer place found; skipping move to place.")

        # 7. Place the dice in the drawer
        try:
            obs, reward, done = execute_place_object(
                env, task, o=dice_to_place, d=drawer_name, p=place_pos if drawer_place is not None else handle_pos
            )
            print(f"[Task] Placed {dice_to_place} in {drawer_name}.")
        except Exception as e:
            print(f"[Error] Could not place {dice_to_place} in {drawer_name}: {e}")
            return

        # 8. Push the drawer closed
        try:
            obs, reward, done = execute_push(
                env, task, d=drawer_name, p=place_pos if drawer_place is not None else handle_pos
            )
            print(f"[Task] Pushed {drawer_name} closed.")
        except Exception as e:
            print(f"[Error] Could not push {drawer_name} closed: {e}")

        # 9. Return to ready pose
        try:
            obs, reward, done = execute_go_ready(
                env, task, from_location=place_pos if drawer_place is not None else handle_pos
            )
            print("[Task] Returned to ready pose.")
        except Exception as e:
            print(f"[Error] Could not return to ready pose: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
