# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from feedback and PDDL) ===
        # Use only objects that are present in the actual object list
        object_names = [
            "dice1", "dice2", "trash", "drawer_top_handle", "drawer_middle_handle", "drawer_bottom_handle"
        ]
        drawer_names = [
            "drawer_top", "drawer_middle", "drawer_bottom"
        ]
        handle_names = [
            "drawer_top_handle", "drawer_middle_handle", "drawer_bottom_handle"
        ]
        drawer_places = [
            "drawer_top_place_left", "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left", "drawer_bottom_place_right"
        ]
        location_names = [
            "floor", "drawer-area", "ready-pose"
        ]
        # For this task, we focus on dice1 and dice2, and the top drawer

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that a predicate such as "lock-known" may be missing.
        # We perform an exploration step to check for missing predicates by using available skills.
        # We use execute_pull to check for lock-known, and execute_go to move, etc.

        # For demonstration, let's try to open the top drawer and put dice1 and dice2 inside.

        # Step 1: Move to ready-pose if not already there
        try:
            robot_pos = positions.get('robot', None)
            if robot_pos is not None and robot_pos != 'ready-pose':
                obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location='ready-pose')
        except Exception as e:
            print("[Exploration] Error moving to ready-pose:", e)

        # Step 2: Pick up the top drawer handle
        try:
            obs, reward, done = execute_pick(
                env, task,
                obj='drawer_top_handle',
                location='ready-pose'
            )
        except Exception as e:
            print("[Exploration] Error picking up drawer_top_handle:", e)

        # Step 3: Pull the top drawer open (this may also reveal lock-known predicate)
        try:
            obs, reward, done = execute_pull(
                env, task,
                drawer='drawer_top',
                handle='drawer_top_handle',
                location='ready-pose'
            )
        except Exception as e:
            print("[Exploration] Error pulling drawer_top:", e)

        # Step 4: Go to dice1 location and pick it up
        try:
            dice1_pos = positions.get('dice1', 'floor')
            obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location=dice1_pos)
            obs, reward, done = execute_pick(
                env, task,
                obj='dice1',
                location=dice1_pos
            )
        except Exception as e:
            print("[Exploration] Error picking up dice1:", e)

        # Step 5: Go to drawer-area and place dice1 in the top drawer
        try:
            obs, reward, done = execute_go(env, task, from_location=dice1_pos, to_location='drawer-area')
            obs, reward, done = execute_place(
                env, task,
                obj='dice1',
                drawer='drawer_top',
                location='drawer-area'
            )
        except Exception as e:
            print("[Exploration] Error placing dice1 in drawer_top:", e)

        # Step 6: Return to ready-pose (if needed)
        try:
            obs, reward, done = execute_go(env, task, from_location='drawer-area', to_location='ready-pose')
        except Exception as e:
            print("[Exploration] Error returning to ready-pose after dice1:", e)

        # Step 7: Repeat for dice2
        try:
            dice2_pos = positions.get('dice2', 'floor')
            obs, reward, done = execute_go(env, task, from_location='ready-pose', to_location=dice2_pos)
            obs, reward, done = execute_pick(
                env, task,
                obj='dice2',
                location=dice2_pos
            )
            obs, reward, done = execute_go(env, task, from_location=dice2_pos, to_location='drawer-area')
            obs, reward, done = execute_place(
                env, task,
                obj='dice2',
                drawer='drawer_top',
                location='drawer-area'
            )
        except Exception as e:
            print("[Exploration] Error handling dice2:", e)

        # Step 8: Close the top drawer
        try:
            obs, reward, done = execute_push(
                env, task,
                drawer='drawer_top',
                location='drawer-area'
            )
        except Exception as e:
            print("[Exploration] Error closing drawer_top:", e)

        # Step 9: Return to ready-pose
        try:
            obs, reward, done = execute_go(env, task, from_location='drawer-area', to_location='ready-pose')
        except Exception as e:
            print("[Exploration] Error returning to ready-pose at end:", e)

        print("[Exploration] Task sequence completed. If any step failed, check for missing predicates such as 'lock-known' or others in the domain.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()