# run_skeleton_task.py (Completed for Exploration Phase – Using Only Provided Skills and Control Flow)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)
        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # -- Exploration Phase: Search for "robot-at ready-pose" Predicate --

        # 1. Get all known locations in the environment
        known_locations = []
        for k, pos in positions.items():
            if "location" in k or "pose" in k or "ready" in k:
                known_locations.append(k)
        # Fallback: If above fails, try several heuristics
        if len(known_locations) == 0:
            for k in positions.keys():
                if "ready" in k or "home" in k:
                    known_locations.append(k)
        # Backup: Use plausible names
        plausible_ready_pose_names = ["ready-pose", "ready_pose", "ready", "home", "orig", "origin"]
        found_ready_pose = None
        for name in plausible_ready_pose_names:
            if name in positions:
                known_locations.append(name)
                found_ready_pose = name
                break
        # Ensure at least one plausible value
        if not found_ready_pose:
            # Use first plausible location
            if "ready-pose" in known_locations:
                found_ready_pose = "ready-pose"
            elif len(known_locations) > 0:
                found_ready_pose = known_locations[0]
            else:
                # fallback: use a hardcoded string
                found_ready_pose = "ready-pose"

        print(f"[Exploration] Using candidate ready-pose: {found_ready_pose}")

        # 2. Find current robot position, and all other locations if possible
        current_robot_location = None
        try:
            # The positions dict may include keys like 'robot', 'robot-at', etc.
            if 'robot' in positions:
                if isinstance(positions['robot'], dict) and 'location' in positions['robot']:
                    current_robot_location = positions['robot']['location']
                elif isinstance(positions['robot'], str):
                    current_robot_location = positions['robot']
            else:
                # Try to infer from observation or env
                current_robot_location = found_ready_pose
        except Exception:
            current_robot_location = found_ready_pose

        # 3. Now, try to execute the skill: move the robot to "ready-pose" using execute_go
        print(f"[Exploration] Moving robot to {found_ready_pose} using execute_go.")

        try:
            # Try to find another location to move from
            all_location_candidates = [loc for loc in positions if loc != found_ready_pose]
            if current_robot_location == found_ready_pose:
                print(f"[Exploration] Robot is already at {found_ready_pose}.")
            else:
                # Use execute_go (from, to):
                from_location = current_robot_location or all_location_candidates[0] if all_location_candidates else found_ready_pose
                to_location = found_ready_pose

                obs, reward, done, info = execute_go(
                    env,
                    task,
                    from_location,
                    to_location
                )
                print(f"[Exploration] Executed execute_go from {from_location} to {to_location}.")
        except Exception as e:
            print(f"[Exploration] Exception in execute_go: {e}")

        # 4. Test if moving to ready-pose enables further skills blocked by 'need-ready'
        print(f"[Exploration] Attempting to perform skills that require 'not (need-ready)'...")

        skills_to_test = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_gripper']
        for skill_name in skills_to_test:
            try:
                skill_func = globals().get(skill_name, None)
                if skill_func is None:
                    continue
                # Try calling the skill with dummy/plausible arguments
                if skill_name == 'execute_gripper':
                    obs, reward, done, info = skill_func(env, task)
                    print(f"[Exploration] Executed {skill_name}")
                elif skill_name == 'execute_sweep':
                    # Need an object and location; try to get from positions
                    object_candidates = [k for k in positions.keys() if "obj" in k or "ball" in k or "handle" in k or "box" in k]
                    if object_candidates:
                        target_obj = object_candidates[0]
                    else:
                        target_obj = list(positions.keys())[0]
                    obs, reward, done, info = skill_func(env, task, target_obj, found_ready_pose)
                    print(f"[Exploration] Executed {skill_name}({target_obj},{found_ready_pose})")
                elif skill_name == 'execute_push':
                    # Need drawer (as per PDDL) and location
                    drawer_candidates = [k for k in positions.keys() if "drawer" in k]
                    if drawer_candidates:
                        drawer = drawer_candidates[0]
                        obs, reward, done, info = skill_func(env, task, drawer, found_ready_pose)
                        print(f"[Exploration] Executed {skill_name}({drawer},{found_ready_pose})")
                elif skill_name == 'execute_pull':
                    drawer_candidates = [k for k in positions.keys() if "drawer" in k]
                    object_candidates = [k for k in positions.keys() if "handle" in k]
                    if drawer_candidates and object_candidates:
                        drawer = drawer_candidates[0]
                        handle = object_candidates[0]
                        obs, reward, done, info = skill_func(env, task, drawer, handle, found_ready_pose)
                        print(f"[Exploration] Executed {skill_name}({drawer},{handle},{found_ready_pose})")
                elif skill_name == 'execute_pick':
                    object_candidates = [k for k in positions.keys() if "obj" in k or "handle" in k or "ball" in k or "box" in k]
                    if object_candidates:
                        target_obj = object_candidates[0]
                    else:
                        target_obj = list(positions.keys())[0]
                    obs, reward, done, info = skill_func(env, task, target_obj, found_ready_pose)
                    print(f"[Exploration] Executed {skill_name}({target_obj},{found_ready_pose})")
                elif skill_name == 'execute_place':
                    object_candidates = [k for k in positions.keys() if "obj" in k or "ball" in k or "box" in k]
                    drawer_candidates = [k for k in positions.keys() if "drawer" in k]
                    if object_candidates and drawer_candidates:
                        target_obj = object_candidates[0]
                        drawer = drawer_candidates[0]
                        obs, reward, done, info = skill_func(env, task, target_obj, drawer, found_ready_pose)
                        print(f"[Exploration] Executed {skill_name}({target_obj},{drawer},{found_ready_pose})")
                elif skill_name == 'execute_rotate':
                    # rotate: treat as gripper for test, ignore params
                    obs, reward, done, info = skill_func(env, task)
                    print(f"[Exploration] Executed {skill_name}")
            except Exception as e:
                print(f"[Exploration] Exception in {skill_name}: {e}")

        # 5. Finished Exploration.
        print("[Exploration] Exploration phase complete. Check for new predicates or skill enablement (such as robot-at ready-pose).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()