# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all predefined skills exactly as provided

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration phase for discovering missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions is expected to be a dict of object_name: (x, y, z) or similar

        # ------------------------------------------------------------------------
        # EXPLORATION PHASE: Identify missing predicate(s)
        # ------------------------------------------------------------------------
        print("[Exploration Phase] Attempting exploration actions to reveal missing predicates...")

        exploration_skills = [
            # As per the exploration knowledge for missing predicates.
            'execute_go_identify', 'execute_go_temperature', 'execute_pick_weight',
            'execute_pick_durability', 'execute_pull'
        ]
        # Provided available skills: 
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Map skill names to functions only if they exist in both knowledge and what you can actually call
        skill_func_map = {}
        for skill_name in available_skills:
            try:
                # Get function from skill_code (already imported *)
                skill_func_map[skill_name] = globals()[skill_name]
            except Exception:
                # Skip if not defined (should not happen)
                continue

        # Exploration heuristic: Try invoking all available skills in plausible ways to determine what's missing.
        # This should trigger any missing predicate errors or feedback.

        objects = list(positions.keys())
        locations = set()
        for v in positions.values():
            if isinstance(v, tuple) or isinstance(v, list):
                locations.add(tuple(v))
        locations = list(locations)
        if len(locations) == 0 and len(objects) > 0:
            # Fallback: just treat the origin as location
            locations = [(0, 0, 0)]

        # For skill functions that might need locations or objects
        def try_skill(skill_func, *args, **kwargs):
            try:
                obs, reward, done = skill_func(env, task, *args, **kwargs)
                print(f"[Exploration] Skill {skill_func.__name__} executed with arguments {args} | Done: {done}")
            except Exception as ex:
                print(f"[Exploration] Skill {skill_func.__name__} FAILED with arguments {args}: {ex}")

        # Try the most basic go operation between two locations
        if len(locations) >= 2 and 'execute_go' in skill_func_map:
            loc_from = locations[0]
            loc_to = locations[1]
            try_skill(skill_func_map['execute_go'], from_location=loc_from, to_location=loc_to)

        # If there is an object, attempt to pick it up and place it. Use the first object detected.
        if len(objects) > 0:
            obj = objects[0]
            if 'execute_pick' in skill_func_map:
                try_skill(skill_func_map['execute_pick'], obj=obj, location=positions[obj])
            # If place is available, try placing in a drawer if any drawer is known
            drawer = None
            for o in objects:
                if 'drawer' in o:
                    drawer = o
                    break
            if drawer and 'execute_place' in skill_func_map:
                try_skill(skill_func_map['execute_place'], obj=obj, drawer=drawer, location=positions[obj])

        # Test pull and push actions if available
        # For pull, requires a drawer and a handle object
        handle = None
        drawer = None
        for o in objects:
            if 'handle' in o:
                handle = o
            if 'drawer' in o:
                drawer = o
        location = positions[handle] if handle else (0, 0, 0)
        if drawer and handle and 'execute_pull' in skill_func_map:
            try_skill(
                skill_func_map['execute_pull'], drawer=drawer, handle=handle, location=location
            )
        if drawer and 'execute_push' in skill_func_map:
            try_skill(skill_func_map['execute_push'], drawer=drawer, location=positions[drawer])

        # Try other skills if available
        if obj and 'execute_sweep' in skill_func_map:
            try_skill(skill_func_map['execute_sweep'], obj=obj, location=positions[obj])
        if 'execute_gripper' in skill_func_map:
            try_skill(skill_func_map['execute_gripper'])

        print("[Exploration Phase] Completed exploratory execution. Review output to identify any missing predicates or skills based on raised errors or environment feedback.")

        # ------------------------------------------------------------------------
        # END EXPLORATION PHASE
        # ------------------------------------------------------------------------

        # === (Insert Plan Execution Code Here for Oracle Plan, if known) ===
        # At this point, you would normally parse and execute your oracle/solution plan step-by-step,
        # invoking the required skills via skill_func_map[obj_action](...) with correct parameters.
        # For this task, only exploration for missing predicates is required per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
