# run_skeleton_task.py (with Exploration Phase for Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use skills as provided, do not redefine

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Debug] Retrieved object positions:", positions)

        # Get list of objects, drawers, locations from positions (assuming typical RLBench schemes)
        objects = [name for name, info in positions.items() if info.get('type', '').lower() == 'object' and not info.get('is_handle', False)]
        handles = [name for name, info in positions.items() if info.get('is_handle', False)]
        drawers = [name for name, info in positions.items() if info.get('type', '').lower() == 'drawer']
        locations = list(set(info['location'] for info in positions.values() if 'location' in info))
        
        # Might want to get the robot's current location as well
        robot_location = None
        for name, info in positions.items():
            if info.get('type', '') == 'robot':
                robot_location = info.get('location', None)
        if robot_location is None and len(locations) > 0:
            robot_location = locations[0]  # Fallback if type information missing

        # === EXPLORATION PHASE: Try to discover missing predicates ===
        print("[Exploration] Starting exploration to discover missing predicates.")
        # Try the skill set to see what fails, to infer missing predicates
        # This example will go through each available skill and attempt logical steps

        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
                            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Track what skills we've tried and their results
        exploration_results = {}

        for skill in available_skills:
            try:
                if skill == 'execute_go':
                    # Try moving the robot between two valid locations
                    if len(locations) >= 2:
                        from_loc = locations[0]
                        to_loc = locations[1]
                        print("[Exploration] Trying execute_go from {} to {}.".format(from_loc, to_loc))
                        # execute_go returns obs, reward, done or similar signature
                        result = execute_go(env, task, from_pos=from_loc, to_pos=to_loc)
                        exploration_results[skill] = 'success'
                    else:
                        exploration_results[skill] = 'skipped (not enough locations)'
                elif skill == 'execute_pick':
                    # Try picking up an object (non-handle)
                    if len(objects) > 0:
                        obj = objects[0]
                        # Pick at the object's location
                        loc = positions[obj]['location']
                        print("[Exploration] Trying execute_pick on object {} at location {}.".format(obj, loc))
                        result = execute_pick(env, task, obj_name=obj, location=loc)
                        exploration_results[skill] = 'success'
                    else:
                        exploration_results[skill] = 'skipped (no objects)'
                elif skill == 'execute_pull':
                    # Try pulling a drawer handle if available
                    if len(handles) > 0 and len(drawers) > 0:
                        handle = handles[0]
                        drawer = drawers[0]
                        loc = positions[handle]['location']
                        print("[Exploration] Trying to pick handle {} and execute_pull on drawer {}.".format(handle, drawer))
                        # First, pick the handle (assume that is needed)
                        try:
                            execute_pick(env, task, obj_name=handle, location=loc)
                        except Exception as e:
                            print("[Exploration][Warning] Picking handle failed:", e)
                        try:
                            result = execute_pull(env, task, drawer_name=drawer, handle_name=handle, location=loc)
                            exploration_results[skill] = 'success'
                        except Exception as e:
                            print("[Exploration][Error] execute_pull failed:", e)
                            exploration_results[skill] = 'fail: ' + str(e)
                    else:
                        exploration_results[skill] = 'skipped (no handles/drawers)'
                elif skill == 'execute_push':
                    # Try to push a random drawer (it must be open)
                    if len(drawers) > 0:
                        drawer = drawers[0]
                        loc = positions[drawer]['location']
                        print("[Exploration] Trying execute_push on drawer {} at location {}.".format(drawer, loc))
                        result = execute_push(env, task, drawer_name=drawer, location=loc)
                        exploration_results[skill] = 'success'
                    else:
                        exploration_results[skill] = 'skipped (no drawers)'
                elif skill == 'execute_place':
                    # Try to place an object in a drawer
                    if len(objects) > 0 and len(drawers) > 0:
                        obj = objects[0]
                        drawer = drawers[0]
                        loc = positions[drawer]['location']
                        print("[Exploration] Trying to place object {} into drawer {} at {}.".format(obj, drawer, loc))
                        result = execute_place(env, task, obj_name=obj, drawer_name=drawer, location=loc)
                        exploration_results[skill] = 'success'
                    else:
                        exploration_results[skill] = 'skipped (no objects or drawers)'
                elif skill == 'execute_sweep':
                    # Optional, try sweeping an object
                    if len(objects) > 0:
                        obj = objects[0]
                        loc = positions[obj]['location']
                        print("[Exploration] Trying execute_sweep on object {} at {}.".format(obj, loc))
                        result = execute_sweep(env, task, obj_name=obj, location=loc)
                        exploration_results[skill] = 'success'
                    else:
                        exploration_results[skill] = 'skipped (no objects)'
                elif skill == 'execute_rotate':
                    # Optional, try rotating if applicable (handle, object)
                    print("[Exploration] Trying execute_rotate (no parameters).")
                    result = execute_rotate(env, task)
                    exploration_results[skill] = 'success'
                elif skill == 'execute_gripper':
                    # Try the gripper
                    print("[Exploration] Trying execute_gripper (no parameters).")
                    result = execute_gripper(env, task)
                    exploration_results[skill] = 'success'
            except Exception as e:
                print("[Exploration][Error] Skill {} failed: {}".format(skill, e))
                exploration_results[skill] = 'fail: ' + str(e)

        print("\n[Exploration Results]")
        for skill, status in exploration_results.items():
            print("  {}: {}".format(skill, status))
        
        print("[Exploration] End of exploration phase. Check errors above for missing predicates or types in failed skills.")
        
        # === Here, the oracle plan should be loaded and executed ===
        # Example placeholder: You would load your plan and step through it here, calling only the predefined skills.

        # For demonstration, END after exploration. 
        # Real system: Replace with code that loads and iterates over an oracle plan, mapping action names to these skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
