# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Location Names (from feedback and PDDL) ===
        dice_name = 'dice1'
        drawer_name = 'drawer_bottom'
        handle_name = 'handle_bottom'
        ready_pose = 'ready-pose'
        floor = 'floor'
        drawer_area = 'drawer-area'

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests that the robot may not know the lock state of the drawer.
        # We will attempt to pull the handle to check if the drawer is locked/unlocked.
        # If the drawer is locked, the pull will fail or not open the drawer.
        # We use execute_pull to "explore" the lock state.

        # Step 1: Move to ready-pose if not already there
        try:
            robot_pos = positions.get('robot', ready_pose)
            if robot_pos != ready_pose:
                obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location=ready_pose)
        except Exception as e:
            print("[Exploration] Could not determine robot position, assuming at ready-pose.")

        # Step 2: Pick up the handle to try to pull the drawer (explore lock state)
        try:
            # Move to handle location if needed
            handle_pos = positions.get(handle_name, floor)
            obs, reward, done = execute_go(env, task, from_location=ready_pose, to_location=handle_pos)
        except Exception as e:
            print(f"[Exploration] Could not move to handle location: {e}")

        try:
            # Pick up the handle
            obs, reward, done = execute_pick_handle(env, task, h=handle_name, p=handle_pos)
        except Exception as e:
            print(f"[Exploration] Could not pick up handle: {e}")

        # Try to pull the drawer to check lock state
        try:
            obs, reward, done = execute_pull(env, task, d=drawer_name, h=handle_name, p=handle_pos)
            print("[Exploration] Pulled the drawer to check lock state.")
        except Exception as e:
            print(f"[Exploration] Could not pull the drawer (may be locked): {e}")

        # === Main Task Plan: Place Dice in Drawer ===

        # Step 3: Move to dice location
        try:
            dice_pos = positions.get(dice_name, floor)
            obs, reward, done = execute_go(env, task, from_location=handle_pos, to_location=dice_pos)
        except Exception as e:
            print(f"[Task] Could not move to dice location: {e}")

        # Step 4: Pick up the dice
        try:
            obs, reward, done = execute_pick_object(env, task, o=dice_name, p=dice_pos)
        except Exception as e:
            print(f"[Task] Could not pick up dice: {e}")

        # Step 5: Return to ready-pose (required by need-ready flag)
        try:
            obs, reward, done = execute_go_ready(env, task, from_location=dice_pos)
        except Exception as e:
            print(f"[Task] Could not return to ready-pose after picking dice: {e}")

        # Step 6: Move to drawer area
        try:
            obs, reward, done = execute_go(env, task, from_location=ready_pose, to_location=drawer_area)
        except Exception as e:
            print(f"[Task] Could not move to drawer area: {e}")

        # Step 7: Place the dice in the drawer (drawer must be open and not full)
        try:
            obs, reward, done = execute_place_object(env, task, o=dice_name, d=drawer_name, p=drawer_area)
        except Exception as e:
            print(f"[Task] Could not place dice in drawer: {e}")

        # Step 8: Return to ready-pose (required by need-ready flag)
        try:
            obs, reward, done = execute_go_ready(env, task, from_location=drawer_area)
        except Exception as e:
            print(f"[Task] Could not return to ready-pose after placing dice: {e}")

        # Step 9: Push the drawer closed
        try:
            obs, reward, done = execute_push(env, task, d=drawer_name, p=drawer_area)
        except Exception as e:
            print(f"[Task] Could not push drawer closed: {e}")

        print("[Task] Finished placing dice in drawer and closing it.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()