# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Robust Object List and Detection ===
        # Define the object list (should match environment objects)
        object_list = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "dice1",
            "dice2",
            "trash",
            "trash_bin"
        ]

        # Example: The task is to insert dice1 into a drawer
        object_to_insert = "dice1"
        target_drawer = "drawer_middle"
        target_handle = "drawer_middle_handle"
        target_location = "drawer_middle_place"
        robot_ready_pose = "ready-pose"

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests we need to robustly check for object presence and type
        # We'll use the available skills to explore and identify object properties

        # 1. Check if the object is in the object list
        if object_to_insert not in object_list:
            print(f"[Error] {object_to_insert} not in object list. Aborting task.")
            return

        # 2. Check if the object is present in the environment (positions)
        if object_to_insert not in positions:
            print(f"[Error] {object_to_insert} not found in environment positions. Aborting task.")
            return

        # 3. Exploration: Is the object a handle? (missing predicate: is-handle)
        # We need to determine if dice1 is a handle or not, as the plan depends on this
        # We'll use the execute_sweep skill as a generic exploration action if available
        # But since the domain has is-handle as a predicate, and the feedback says (is-handle dice1) is missing,
        # we simulate an exploration step to check this

        is_handle = False
        try:
            # Try to use execute_pick_handle (if it fails, it's not a handle)
            obs, reward, done = execute_pick_handle(
                env,
                task,
                obj=object_to_insert,
                p=positions[object_to_insert]
            )
            if done:
                is_handle = True
                print(f"[Exploration] {object_to_insert} is a handle (is-handle true).")
            else:
                is_handle = False
        except Exception as e:
            # If the skill fails, it's not a handle
            is_handle = False
            print(f"[Exploration] {object_to_insert} is not a handle (is-handle false). Exception: {e}")

        # If the object is not a handle, proceed with object pick
        if not is_handle:
            # 4. Verify the robot is at the correct location for picking
            robot_location = None
            for loc in positions:
                if "robot" in loc or "ready" in loc:
                    robot_location = loc
                    break
            if robot_location is None:
                robot_location = robot_ready_pose

            # Move robot to the object's location if needed
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=positions[object_to_insert]
                )
                print(f"[Action] Robot moved to {object_to_insert} location.")
            except Exception as e:
                print(f"[Warning] Could not move robot to {object_to_insert} location: {e}")

            # 5. Pick the object
            try:
                obs, reward, done = execute_pick_object(
                    env,
                    task,
                    o=object_to_insert,
                    p=positions[object_to_insert]
                )
                print(f"[Action] Picked up {object_to_insert}.")
            except Exception as e:
                print(f"[Error] Failed to pick up {object_to_insert}: {e}")
                return

            # 6. Move to drawer location
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions[object_to_insert],
                    to_location=positions[target_drawer] if target_drawer in positions else positions[target_location]
                )
                print(f"[Action] Robot moved to drawer location.")
            except Exception as e:
                print(f"[Warning] Could not move robot to drawer location: {e}")

            # 7. Place the object in the drawer
            try:
                obs, reward, done = execute_place_object(
                    env,
                    task,
                    o=object_to_insert,
                    d=target_drawer,
                    p=positions[target_drawer] if target_drawer in positions else positions[target_location]
                )
                print(f"[Action] Placed {object_to_insert} in {target_drawer}.")
            except Exception as e:
                print(f"[Error] Failed to place {object_to_insert} in {target_drawer}: {e}")
                return

            # 8. Return to ready pose
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=positions[target_drawer] if target_drawer in positions else positions[target_location]
                )
                print(f"[Action] Robot returned to ready pose.")
            except Exception as e:
                print(f"[Warning] Could not return to ready pose: {e}")

        else:
            # If the object is a handle, use handle-specific actions
            print(f"[Info] {object_to_insert} is a handle. Executing handle-specific plan.")
            # Move to handle location
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_ready_pose,
                    to_location=positions[object_to_insert]
                )
                print(f"[Action] Robot moved to handle location.")
            except Exception as e:
                print(f"[Warning] Could not move robot to handle location: {e}")

            # Pick the handle
            try:
                obs, reward, done = execute_pick_handle(
                    env,
                    task,
                    h=object_to_insert,
                    p=positions[object_to_insert]
                )
                print(f"[Action] Picked up handle {object_to_insert}.")
            except Exception as e:
                print(f"[Error] Failed to pick up handle {object_to_insert}: {e}")
                return

            # Pull the drawer using the handle
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    d=target_drawer,
                    h=object_to_insert,
                    p=positions[target_drawer] if target_drawer in positions else positions[object_to_insert]
                )
                print(f"[Action] Pulled drawer {target_drawer} using handle {object_to_insert}.")
            except Exception as e:
                print(f"[Error] Failed to pull drawer {target_drawer}: {e}")
                return

            # Return to ready pose
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=positions[target_drawer] if target_drawer in positions else positions[object_to_insert]
                )
                print(f"[Action] Robot returned to ready pose.")
            except Exception as e:
                print(f"[Warning] Could not return to ready pose: {e}")

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
