# run_skeleton_task.py (Completed with Exploration and Predicate Check)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Define object list for drawer placement ===
        # This should be provided or inferred from the environment/observation
        # For demonstration, we assume object_list is available from the environment or task
        try:
            object_list = task.object_list  # Try to get from task if available
        except AttributeError:
            # Fallback: try to infer from positions or observation
            object_list = [obj for obj in positions.keys() if "object" in obj or "obj" in obj]
        print("[Info] object_list for drawer placement:", object_list)

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests we need to ensure the object is in object_list before placing in drawer.
        # We also need to check for 'identified' predicate (from exploration domain) before manipulation.
        # We'll simulate an exploration step to identify all objects at their locations.

        # For each object, ensure it is identified before manipulation
        identified_objects = set()
        for obj_name in object_list:
            obj_pos = positions.get(obj_name, None)
            if obj_pos is None:
                print(f"[Warning] Object {obj_name} position not found, skipping identification.")
                continue
            # Simulate identification (e.g., vision or sensor check)
            try:
                # Use execute_go skill to move to the object's location if not already there
                robot_current_pos = task.get_robot_position()
                if robot_current_pos != obj_pos:
                    obs, reward, done = execute_go(env, task, from_location=robot_current_pos, to_location=obj_pos)
                    if done:
                        print(f"[Exploration] Robot moved to {obj_name} at {obj_pos}.")
                # Simulate identification (could be a vision check or a no-op)
                # In a real system, you might call a vision or sensor skill here
                identified_objects.add(obj_name)
                print(f"[Exploration] Identified object: {obj_name}")
            except Exception as e:
                print(f"[Error] Failed to identify {obj_name}: {e}")

        # === Main Task Plan Execution ===
        # Example: Pick up each object and place it in the drawer if allowed
        # For demonstration, we assume a single drawer and its position
        drawer_name = None
        drawer_pos = None
        for name in positions:
            if "drawer" in name:
                drawer_name = name
                drawer_pos = positions[name]
                break
        if drawer_name is None or drawer_pos is None:
            print("[Error] No drawer found in positions. Aborting task.")
            return

        # For demonstration, we assume the robot starts at a 'ready-pose' or initial position
        robot_pos = task.get_robot_position()

        # Open the drawer if needed (simulate pull with handle if required)
        # Find handle for the drawer if present
        handle_name = None
        for name in positions:
            if "handle" in name and drawer_name in name:
                handle_name = name
                break

        # If handle exists, pick handle and pull to open drawer
        if handle_name is not None:
            # Move to handle
            handle_pos = positions[handle_name]
            if robot_pos != handle_pos:
                try:
                    obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location=handle_pos)
                    robot_pos = handle_pos
                except Exception as e:
                    print(f"[Error] Failed to move to handle: {e}")
            # Pick handle
            try:
                obs, reward, done = execute_pick(env, task, object_name=handle_name, location=handle_pos)
                print(f"[Task] Picked handle: {handle_name}")
            except Exception as e:
                print(f"[Error] Failed to pick handle: {e}")
            # Pull to open drawer
            try:
                obs, reward, done = execute_pull(env, task, drawer_name=drawer_name, handle_name=handle_name, location=handle_pos)
                print(f"[Task] Pulled handle to open drawer: {drawer_name}")
            except Exception as e:
                print(f"[Error] Failed to pull handle: {e}")

        # For each object, pick and place in drawer if in object_list
        for obj_name in object_list:
            if obj_name not in identified_objects:
                print(f"[Warning] Object {obj_name} not identified, skipping.")
                continue
            obj_pos = positions.get(obj_name, None)
            if obj_pos is None:
                print(f"[Warning] Object {obj_name} position not found, skipping.")
                continue
            # Move to object
            if robot_pos != obj_pos:
                try:
                    obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location=obj_pos)
                    robot_pos = obj_pos
                except Exception as e:
                    print(f"[Error] Failed to move to {obj_name}: {e}")
            # Pick object
            try:
                obs, reward, done = execute_pick(env, task, object_name=obj_name, location=obj_pos)
                print(f"[Task] Picked object: {obj_name}")
            except Exception as e:
                print(f"[Error] Failed to pick {obj_name}: {e}")
                continue
            # Move to drawer
            if robot_pos != drawer_pos:
                try:
                    obs, reward, done = execute_go(env, task, from_location=robot_pos, to_location=drawer_pos)
                    robot_pos = drawer_pos
                except Exception as e:
                    print(f"[Error] Failed to move to drawer: {e}")
            # Check if object is in object_list before placing
            if obj_name not in object_list:
                print(f"[Error] {obj_name} is not in object_list, cannot place in drawer.")
                continue
            # Place object in drawer
            try:
                obs, reward, done = execute_place(env, task, object_name=obj_name, drawer_name=drawer_name, location=drawer_pos)
                print(f"[Task] Placed {obj_name} in {drawer_name}")
            except Exception as e:
                print(f"[Error] Failed to place {obj_name} in {drawer_name}: {e}")
            # After placing, return to ready pose if required
            try:
                obs, reward, done = execute_go_ready(env, task, from_location=drawer_pos)
                robot_pos = 'ready-pose'
                print(f"[Task] Returned to ready pose after placing {obj_name}")
            except Exception as e:
                print(f"[Warning] Could not return to ready pose: {e}")

        # Optionally, close the drawer if required
        try:
            obs, reward, done = execute_push(env, task, drawer_name=drawer_name, location=drawer_pos)
            print(f"[Task] Closed drawer: {drawer_name}")
        except Exception as e:
            print(f"[Warning] Could not close drawer: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
