# run_skeleton_task.py (Completed Executable Script)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # (Predefined primitives only -- do not redefine.)
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video recording
        init_video_writers(obs)

        # Wrap task steps for video/observation capture
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Diagnose Missing Predicate (e.g., need-ready) ===
        print("[Exploration] Checking for need-ready predicate requirements...")

        # Many primitive actions in this domain require "(not (need-ready))" as a precondition.
        # The need-ready predicate is set to True after some skill calls (like execute_pick_object/place_object)
        # and must be explicitly cleared (using execute_go_ready) before further actions.

        # Let's attempt a pick-place sequence to explore and check precondition enforcement.

        # Find one object (not a handle) on the floor and one drawer, handle, and location for demonstration.
        object_name, object_info = None, None
        handle_name, handle_info = None, None
        drawer_name, drawer_info = None, None
        drawer_location = None
        ready_pose_location = "ready-pose"
        robot_location = None

        # Find an object, handle, drawer from `positions` (simulate as needed)
        for name, info in positions.items():
            # Assume info dict has keys: 'type' (object/drawer/handle), 'location'
            ty = info.get('type', '').lower()
            if not object_name and ty == 'object' and not info.get('is_handle', False):
                object_name = name
                object_info = info
            if not handle_name and ty == 'object' and info.get('is_handle', False):
                handle_name = name
                handle_info = info
            if not drawer_name and ty == 'drawer':
                drawer_name = name
                drawer_info = info
            if not robot_location and ty == 'location' and name == 'ready-pose':
                ready_pose_location = name

        if not robot_location:
            # Default to some location found in the positions
            for name, info in positions.items():
                if info.get('type', '').lower() == 'location':
                    robot_location = name
                    break

        if not object_name or not drawer_name:
            print("[ERROR] Could not find suitable object and drawer in positions dictionary.")
            return

        # Try the following sequence:
        # 1. Move to object's location (if not already there)
        object_location = object_info['location']
        robot_current_location = ready_pose_location   # initial assumption

        if robot_current_location != object_location:
            print(f"[Skill] execute_go: Moving robot from {robot_current_location} to {object_location}")
            try:
                obs, reward, done, info = execute_go(
                    env,
                    from_location=robot_current_location,
                    to_location=object_location
                )
                robot_current_location = object_location
            except Exception as e:
                print(f"[ERROR] execute_go failed: {e}")
                return

        # 2. Try to pick the object (triggers need-ready predicate)
        print(f"[Skill] execute_pick: Picking up object '{object_name}' at {object_location}")
        try:
            obs, reward, done, info = execute_pick(
                env,
                object_name,
                object_location
            )
        except Exception as e:
            print(f"[ERROR] execute_pick failed: {e}")
            return

        # At this point, need-ready will be set TRUE; most skills (except go_ready) will now be blocked.

        # 3. Explore: Try to place the object (this should FAIL if need-ready is still set)
        print(f"[Exploration] Trying to execute place immediately (should fail due to need-ready)")
        try:
            obs, reward, done, info = execute_place(
                env,
                object_name,
                drawer_name,
                robot_current_location   # trying to place at current location without go_ready
            )
            print("[WARNING] Place unexpectedly succeeded, but should require clearing need-ready first.")
        except Exception as e:
            print("[Exploration] Place failed as expected (possibly due to need-ready).")
            print(f"  Details: {e}")

        # 4. Clear (need-ready) with execute_go_ready
        print("[Skill] execute_go_ready: Returning robot to ready-pose to clear need-ready")
        try:
            obs, reward, done, info = execute_go_ready(
                env,
                robot_current_location
            )
            robot_current_location = ready_pose_location
        except Exception as e:
            print(f"[ERROR] execute_go_ready failed: {e}")
            return

        # 5. Move to drawer for placing the object (if not already there)
        drawer_location = drawer_info['location']
        if robot_current_location != drawer_location:
            print(f"[Skill] execute_go: Moving robot from {robot_current_location} to {drawer_location}")
            try:
                obs, reward, done, info = execute_go(
                    env,
                    from_location=robot_current_location,
                    to_location=drawer_location
                )
                robot_current_location = drawer_location
            except Exception as e:
                print(f"[ERROR] execute_go failed: {e}")
                return

        # 6. Place the object in the drawer (should now succeed since need-ready is clear)
        print(f"[Skill] execute_place: Placing object '{object_name}' in drawer '{drawer_name}' at {drawer_location}")
        try:
            obs, reward, done, info = execute_place(
                env,
                object_name,
                drawer_name,
                drawer_location
            )
            print("[Result] Place succeeded after clearing need-ready.")
        except Exception as e:
            print(f"[ERROR] execute_place failed: {e}")
            return

        # 7. If desired, repeat pick/place cycles or perform additional planned actions.
        # This flow demonstrates the need to clear the (need-ready) predicate between certain actions.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()