# run_skeleton_task.py (Enhanced with Exploration Phase for Missing Predicate Detection)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only externally defined skills

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with an exploration phase to identify missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()   # e.g., {'obj1':(x,y,z), 'drawer1':...}

        # === Access available objects/locations ===
        # If needed, parse from descriptions/obs/positions
        # For demonstration, attempt to extract some keys if present.
        all_object_names = list(positions.keys())
        drawer_names = [n for n in all_object_names if 'drawer' in n]
        handle_names = [n for n in all_object_names if 'handle' in n]
        location_names = [n for n in all_object_names if 'location' in n or 'pose' in n]  # update as per env

        # Fallbacks for agent position etc.
        robot_init_pos = positions.get('robot', None)  # or get from descriptions/obs

        # === EXPLORATION PHASE: Try to discover missing predicates ===
        print(">> [Exploration Phase] Starting missing predicate discovery...")
        # This will simulate the PDDL exploration knowledge: e.g., gaining 'identified', 'weight-known', etc.

        exploration_success = {}
        # The available exploration-like skills are 'execute_go', 'execute_pick', 'execute_pull', etc.
        # We try to use them in a sequence to see which predicates become true and which not

        try:
            # Move to all known locations (if possible)
            print("[Exploration] Trying moves to all accessible locations...")
            if len(location_names) >= 2:
                # Try moves between all pairs (might help reveal problems with robot-at)
                for i in range(len(location_names)-1):
                    from_loc = location_names[i]
                    to_loc = location_names[i+1]
                    try:
                        print(f"[Exploration] Moving robot from {from_loc} to {to_loc} using execute_go...")
                        obs, reward, done, info = execute_go(
                            env, 
                            task, 
                            from_loc=from_loc, 
                            to_loc=to_loc
                        )
                        print("  (Exploration) Move executed successfully.")
                        exploration_success[f"move_{from_loc}_to_{to_loc}"] = True
                        if done:
                            print("(Exploration) Task unexpectedly finished during move.")
                            break
                    except Exception as e:
                        print(f"  [Exploration] Move from {from_loc} to {to_loc} failed: {e}")
                        exploration_success[f"move_{from_loc}_to_{to_loc}"] = False

            # Try picking up each object from its position
            print("[Exploration] Trying pick actions for each object...")
            for obj_name in all_object_names:
                for loc_name in location_names:
                    try:
                        print(f"[Exploration] Picking {obj_name} at {loc_name} using execute_pick...")
                        obs, reward, done, info = execute_pick(
                            env,
                            task,
                            obj_name,
                            loc_name
                        )
                        print("  (Exploration) Pick executed successfully.")
                        exploration_success[f"pick_{obj_name}_at_{loc_name}"] = True
                        if done:
                            print("(Exploration) Task unexpectedly finished during pick.")
                            break
                    except Exception as e:
                        print(f"  [Exploration] Pick {obj_name} at {loc_name} failed: {e}")
                        exploration_success[f"pick_{obj_name}_at_{loc_name}"] = False

            # Try pulling on each handle (if any)
            print("[Exploration] Trying pull actions for each handle...")
            for h in handle_names:
                for d in drawer_names:
                    for loc_name in location_names:
                        try:
                            print(f"[Exploration] Pulling {d} with {h} at {loc_name} using execute_pull...")
                            obs, reward, done, info = execute_pull(
                                env,
                                task,
                                d,
                                h,
                                loc_name
                            )
                            print("  (Exploration) Pull executed successfully.")
                            exploration_success[f"pull_{d}_{h}_at_{loc_name}"] = True
                            if done:
                                print("(Exploration) Task unexpectedly finished during pull.")
                                break
                        except Exception as e:
                            print(f"  [Exploration] Pull {d} with {h} at {loc_name} failed: {e}")
                            exploration_success[f"pull_{d}_{h}_at_{loc_name}"] = False

            # Try push actions for each drawer
            print("[Exploration] Trying push actions for each drawer...")
            for d in drawer_names:
                for loc_name in location_names:
                    try:
                        print(f"[Exploration] Pushing {d} at {loc_name} using execute_push...")
                        obs, reward, done, info = execute_push(
                            env,
                            task,
                            d,
                            loc_name
                        )
                        print("  (Exploration) Push executed successfully.")
                        exploration_success[f"push_{d}_at_{loc_name}"] = True
                        if done:
                            print("(Exploration) Task unexpectedly finished during push.")
                            break
                    except Exception as e:
                        print(f"  [Exploration] Push {d} at {loc_name} failed: {e}")
                        exploration_success[f"push_{d}_at_{loc_name}"] = False

            # Try other provided skills (sweep, gripper, rotate) generically
            print("[Exploration] Trying 'execute_gripper', 'execute_sweep', 'execute_rotate'...")
            try:
                obs, reward, done, info = execute_gripper(env, task)
                print("[Exploration] execute_gripper succeeded.")
                exploration_success['execute_gripper'] = True
            except Exception as e:
                print(f"[Exploration] execute_gripper failed: {e}")
                exploration_success['execute_gripper'] = False
            for obj_name in all_object_names:
                for loc_name in location_names:
                    try:
                        obs, reward, done, info = execute_sweep(env, task, obj_name, loc_name)
                        print(f"[Exploration] execute_sweep({obj_name}, {loc_name}) succeeded.")
                        exploration_success[f"execute_sweep_{obj_name}_{loc_name}"] = True
                    except Exception as e:
                        print(f"[Exploration] execute_sweep({obj_name}, {loc_name}) failed: {e}")
                        exploration_success[f"execute_sweep_{obj_name}_{loc_name}"] = False
            try:
                obs, reward, done, info = execute_rotate(env, task)
                print("[Exploration] execute_rotate succeeded.")
                exploration_success['execute_rotate'] = True
            except Exception as e:
                print(f"[Exploration] execute_rotate failed: {e}")
                exploration_success['execute_rotate'] = False

            print(">> [Exploration Phase] Complete. Predicate discovery (missing predicates may be inferred from failed actions).")
            print(">>> [Exploration Results] Summary of attempted actions:")
            for k, v in exploration_success.items():
                print(f"    {k}: {'SUCCESS' if v else 'FAIL'}")

        except Exception as e:
            print(f"[Exploration Phase] An unexpected error occurred: {e}")

        # === END OF EXPLORATION ===
        # (At this point, external tools may use the exploration results to identify missing predicates to add to the domain/problem.)

        # === (Optional) Proceed to main task plan execution ===
        print(">> [Task Execution Phase] No specific plan implemented after exploration (this is generic).")
        # You would add your actual oracle plan code here, similar to the above, using appropriate skill invocations.

        # TODO: If an oracle plan is available, implement it below using only the provided skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()