# run_skeleton_task.py (Completed according to requirements)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills ONLY

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation,
       extended for exploration to detect missing predicate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Exploration Phase: Try the skill set to probe environment and find missing predicate
        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Collect names for debug
        print("[Exploration] Available skills:", available_skills)
        # Try each skill in a safe and generic way to observe behavior and environment feedback
        # We only call *imported* skills; do not define or implement any new skills

        # Attempt to identify missing predicates by examining common critical action points:
        # We'll use 'positions' to interact with one or two objects if possible

        # Get example object/location/possibly handle names (be robust to missing names in positions)
        obj_names = [name for name in positions if not name.startswith('drawer') and not name.startswith('handle')]
        drawer_names = [name for name in positions if name.startswith('drawer')]
        handle_names = [name for name in positions if name.startswith('handle')]
        location_names = list(set([v['location'] if isinstance(v, dict) and 'location' in v else None for v in positions.values()]))
        # Remove None from location_names
        location_names = [l for l in location_names if l]

        # Simple debugging report
        print("[Exploration] Objects:", obj_names)
        print("[Exploration] Drawers:", drawer_names)
        print("[Exploration] Handles:", handle_names)
        print("[Exploration] Locations:", location_names)

        # Sanity checks
        if not obj_names or not drawer_names or not location_names:
            print("[Exploration] Warning: Not enough objects/drawers/locations to explore.")
        else:
            # Pick an object, a drawer, a handle, and two locations
            test_obj = obj_names[0]
            test_drawer = drawer_names[0]
            test_handle = handle_names[0] if handle_names else None
            from_location = location_names[0]
            to_location = location_names[1] if len(location_names) > 1 else location_names[0]

            print(f"[Exploration] Will test pick/place on: {test_obj}, {test_drawer}, {test_handle}, {from_location}, {to_location}")

            # 1. Try to move the robot to a location using execute_go
            try:
                print(f"[Exploration] Trying execute_go: moving robot from {from_location} to {to_location}")
                obs, reward, done, info = execute_go(env, task, from_location, to_location)
                print("[Exploration] execute_go result:", done, info)
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

            # 2. Try picking an object (if available)
            try:
                print(f"[Exploration] Trying execute_pick: picking up {test_obj} at {to_location}")
                obs, reward, done, info = execute_pick(env, task, test_obj, to_location)
                print("[Exploration] execute_pick result:", done, info)
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

            # 3. If there is a drawer and handle, try pull (expects handle to be held)
            if test_handle:
                try:
                    print(f"[Exploration] Trying execute_pick: picking up handle {test_handle} at {to_location}")
                    obs, reward, done, info = execute_pick(env, task, test_handle, to_location)
                    print("[Exploration] execute_pick handle result:", done, info)
                except Exception as e:
                    print(f"[Exploration] execute_pick on handle failed: {e}")

                try:
                    print(f"[Exploration] Trying execute_pull: pulling drawer {test_drawer} using handle {test_handle} at {to_location}")
                    obs, reward, done, info = execute_pull(env, task, test_drawer, test_handle, to_location)
                    print("[Exploration] execute_pull result:", done, info)
                except Exception as e:
                    print(f"[Exploration] execute_pull failed: {e}")

            # 4. Try place (if robot is holding test_obj)
            try:
                print(f"[Exploration] Trying execute_place: placing object {test_obj} into drawer {test_drawer} at {to_location}")
                obs, reward, done, info = execute_place(env, task, test_obj, test_drawer, to_location)
                print("[Exploration] execute_place result:", done, info)
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

            # 5. Try push the drawer
            try:
                print(f"[Exploration] Trying execute_push: pushing drawer {test_drawer} at {to_location}")
                obs, reward, done, info = execute_push(env, task, test_drawer, to_location)
                print("[Exploration] execute_push result:", done, info)
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

            # 6. Try sweep (on test_obj at to_location)
            try:
                print(f"[Exploration] Trying execute_sweep: sweeping {test_obj} at {to_location}")
                obs, reward, done, info = execute_sweep(env, task, test_obj, to_location)
                print("[Exploration] execute_sweep result:", done, info)
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")

            # 7. Try gripper (no arguments)
            try:
                print(f"[Exploration] Trying execute_gripper")
                obs, reward, done, info = execute_gripper(env, task)
                print("[Exploration] execute_gripper result:", done, info)
            except Exception as e:
                print(f"[Exploration] execute_gripper failed: {e}")

            # 8. Try rotate if skill exists (not in PDDL, so ignore if not present)
            if 'execute_rotate' in available_skills:
                try:
                    print(f"[Exploration] Trying execute_rotate: rotating {test_obj} at {to_location}")
                    obs, reward, done, info = execute_rotate(env, task, test_obj, to_location)
                    print("[Exploration] execute_rotate result:", done, info)
                except Exception as e:
                    print(f"[Exploration] execute_rotate failed: {e}")

        # Further logic: Based on observed output/logging/errors, analyst may deduce missing predicates for planner.
        # (The exploration phase is mainly diagnostic and informative to help human or agent fix domain.)

        # Once exploration is finished: task is done for this diagnostic run

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
