# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # you don't need to redefine primitives like move, pick, place

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # This will help us find drawers, handles, robot locations, etc.
        positions = get_object_positions()

        # For demonstration purposes, let's assume some typical naming conventions.
        # You should adjust these according to your setup or object naming.
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        ready_pose_name = 'ready-pose'
        floor_object_name = 'object1'
        robot_location_name = 'robot_home'  # or initial robot location
        drawer_location_name = 'drawer1_location'

        try:
            # Attempt to get object positions – fallback if missing
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            ready_pose_pos = positions.get(ready_pose_name, None)
            floor_object_pos = positions.get(floor_object_name, None)
            robot_home_pos = positions.get(robot_location_name, None)
            drawer_location_pos = positions.get(drawer_location_name, None)
        except Exception as e:
            print("Error retrieving object positions:", e)
            drawer_pos = handle_pos = ready_pose_pos = floor_object_pos = robot_home_pos = drawer_location_pos = None

        # --- State Tracking for Exploration ---
        exploration_done = False
        missing_predicates_found = []

        # ------ EXPLORATION PHASE ------
        # The feedback indicates that (drawer-unlocked drawer1) is a missing predicate required for action feasibility.
        # We'll perform an exploration to check if the drawer's lock state is known, and if not attempt an action
        # that will provide that information, as suggested by the exploration PDDL.
        # We assume robot can check 'lock-known' on drawer by attempting 'execute_pull' exploration.
        print("[Exploration] Checking if lock state of drawer is known...")
        # Simulated state check (you might access your environment state here)
        drawer_lock_known = False
        # We'll suppose we do not know lock state – in a true implementation you'd query env/task state
        if not drawer_lock_known:
            print("[Exploration] Lock state unknown. Attempting to determine lock status via allowed skills.")

            # Use the exploration skill if available (execute_pull for lock-known).
            # We use execute_pull; must select the handle as the object of interest.
            # If that skill throws an error (e.g. handle not found), handle gracefully.
            # The robot should be at the drawer location for this.
            try:
                # Move robot to drawer location if needed.
                # Use execute_go to move from current location to drawer location.
                # These might be names like 'robot_home' and 'drawer1_location', adjust accordingly.
                print("[Exploration] Moving robot to drawer location for lock check...")
                obs, reward, done = execute_go(env, task, from_location=robot_location_name, to_location=drawer_location_name)
            except Exception as e:
                print("[Exploration] Error moving robot during exploration:", e)

            # Try to ensure robot is holding handle before attempting pull.
            # First, pick up the handle if possible.
            try:
                print("[Exploration] Picking up handle to test lock state...")
                obs, reward, done = execute_pick(env, task, obj=handle_name, location=drawer_location_name)
            except Exception as e:
                print("[Exploration] Error picking handle during exploration:", e)

            # Now, attempt to pull the handle to test the lock status.
            try:
                print("[Exploration] Pulling handle to determine lock status...")
                obs, reward, done = execute_pull(env, task, drawer=drawer_name, handle=handle_name, location=drawer_location_name)
                # The attempt to pull should update lock-known predicate if following the exploration domain logic.
                drawer_lock_known = True
                missing_predicates_found.append('(drawer-unlocked {})'.format(drawer_name))
                print(f"[*] Missing predicate found: (drawer-unlocked {drawer_name})")
            except Exception as e:
                print("[Exploration] Error pulling handle during exploration (may indicate lock):", e)
                # Could add heuristic here to mark as locked or unlocked
                # For now, simply record the error and continue.
                missing_predicates_found.append('(drawer-unlocked {}) [FAILED TO VERIFY]'.format(drawer_name))

            exploration_done = True
        else:
            print("[Exploration] Lock state already known.")

        # ==== MAIN ORACLE PLAN EXECUTION ====
        print("=== Starting Main Oracle Plan Execution ===")

        # 1. Assume after exploration, we know we need (drawer-unlocked drawer1)
        # Continue with plan: open drawer (requires unlocked state)

        # Ensure robot is at drawer, holding handle. Move and pick if needed.
        try:
            print("[Plan] Moving robot to drawer location for main task...")
            obs, reward, done = execute_go(env, task, from_location=robot_location_name, to_location=drawer_location_name)
        except Exception as e:
            print("[Plan] Error in robot move to drawer location:", e)

        try:
            print("[Plan] Picking up drawer handle...")
            obs, reward, done = execute_pick(env, task, obj=handle_name, location=drawer_location_name)
        except Exception as e:
            print("[Plan] Error in picking handle:", e)

        # Attempt to pull the handle (open drawer) -- now with missing predicate satisfied by exploration
        try:
            print("[Plan] Pulling drawer open with handle...")
            obs, reward, done = execute_pull(env, task, drawer=drawer_name, handle=handle_name, location=drawer_location_name)
        except Exception as e:
            print("[Plan] Error pulling drawer (may be still locked):", e)
            # If you reach here, likely a domain/model/run mismatch.

        # Place an object into the drawer, assuming domain/obs allows it
        try:
            print("[Plan] Moving robot to object location...")
            obs, reward, done = execute_go(env, task, from_location=drawer_location_name, to_location=floor_object_name)
        except Exception as e:
            print("[Plan] Error moving robot to object:", e)

        try:
            print("[Plan] Picking up object...")
            obs, reward, done = execute_pick(env, task, obj=floor_object_name, location=floor_object_name)
        except Exception as e:
            print("[Plan] Error picking object:", e)

        try:
            print("[Plan] Moving robot with object to drawer for placement...")
            obs, reward, done = execute_go(env, task, from_location=floor_object_name, to_location=drawer_location_name)
        except Exception as e:
            print("[Plan] Error moving robot for place step:", e)

        try:
            print("[Plan] Placing object into the open drawer...")
            obs, reward, done = execute_place(env, task, obj=floor_object_name, drawer=drawer_name, location=drawer_location_name)
        except Exception as e:
            print("[Plan] Error placing object into drawer:", e)

        # Push drawer closed, if that's part of goal
        try:
            print("[Plan] Pushing drawer closed...")
            obs, reward, done = execute_push(env, task, drawer=drawer_name, location=drawer_location_name)
        except Exception as e:
            print("[Plan] Error pushing drawer closed:", e)

        # Return robot to ready-pose if needed
        try:
            print("[Plan] Returning robot to ready pose...")
            obs, reward, done = execute_go(env, task, from_location=drawer_location_name, to_location=ready_pose_name)
        except Exception as e:
            print("[Plan] Error returning to ready-pose:", e)

        print("=== Main Plan Execution Complete ===")

        if missing_predicates_found:
            print("[Summary] Exploration found missing predicates:")
            for pred in missing_predicates_found:
                print("   ", pred)
        else:
            print("[Summary] No missing predicates found during exploration.")

        print("===== Task Run Complete =====")
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
